<?php
session_start();
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../includes/functions.php';
require_once __DIR__ . '/../../includes/export_helpers.php';

require_login();

// Restrict to Accountant, Admin, DG, DGA, Board roles
$allowed_roles = ['Accountant', 'Admin', 'DG', 'DGA', 'Board'];
if (!in_array($_SESSION['role_name'], $allowed_roles, true)) {
    error_log("Unauthorized access to loan_repayment_history.php by user {$_SESSION['full_name']} ({$_SESSION['role_name']})");
    header("Location: /masunzu_bar_hotel/dashboards/unauthorized.php");
    exit;
}

$is_admin = in_array($_SESSION['role_name'], ['Admin', 'DG', 'DGA', 'Board'], true);

$conn = get_db_connection();
if (!$conn) {
    error_log("Database connection failed in loan_repayment_history.php: " . mysqli_connect_error());
    http_response_code(500);
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Échec de la connexion à la base de données. Contactez l'administrateur.</p>";
    include __DIR__ . '/../../includes/footer.php';
    exit;
}

// Get province_id (only for accountants, admins see all)
$province_id = null;
if (!$is_admin) {
    $province_id = $_SESSION['province_id'] ?? null;
    if (!$province_id) {
        $stmtProvince = $conn->prepare("SELECT province_id FROM user WHERE id = ?");
        if ($stmtProvince) {
            $stmtProvince->bind_param("i", $_SESSION['user_id']);
            $stmtProvince->execute();
            $provinceRow = $stmtProvince->get_result()->fetch_assoc();
            $_SESSION['province_id'] = $provinceRow['province_id'] ?? null;
            $province_id = $_SESSION['province_id'];
            $stmtProvince->close();
        }
    }
}

// Handle filters
$filter_customer_id = isset($_GET['customer_id']) ? (int)$_GET['customer_id'] : 0;
$filter_date_from = isset($_GET['date_from']) ? trim($_GET['date_from']) : '';
$filter_date_to = isset($_GET['date_to']) ? trim($_GET['date_to']) : '';
$search_term = isset($_GET['search']) ? trim($_GET['search']) : '';

// Handle CSV export
if (isset($_GET['export']) && $_GET['export'] === 'csv') {
    $where_conditions = [];
    $params = [];
    $param_types = '';
    
    if ($filter_customer_id > 0) {
        $where_conditions[] = "lp.customer_id = ?";
        $params[] = $filter_customer_id;
        $param_types .= 'i';
    }
    
    if ($filter_date_from) {
        $where_conditions[] = "DATE(lp.payment_date) >= ?";
        $params[] = $filter_date_from;
        $param_types .= 's';
    }
    
    if ($filter_date_to) {
        $where_conditions[] = "DATE(lp.payment_date) <= ?";
        $params[] = $filter_date_to;
        $param_types .= 's';
    }
    
    if ($search_term) {
        $where_conditions[] = "(c.full_name LIKE ? OR ui.invoice_number LIKE ? OR lp.notes LIKE ?)";
        $searchPattern = '%' . $search_term . '%';
        $params[] = $searchPattern;
        $params[] = $searchPattern;
        $params[] = $searchPattern;
        $param_types .= 'sss';
    }
    
    if ($province_id && !$is_admin) {
        $where_conditions[] = "lp.province_id = ?";
        $params[] = $province_id;
        $param_types .= 'i';
    }
    
    $where_clause = !empty($where_conditions) ? 'WHERE ' . implode(' AND ', $where_conditions) : '';
    
    $export_query = "
        SELECT 
            lp.id,
            lp.payment_date,
            c.full_name AS customer_name,
            c.tel AS customer_phone,
            p.name AS province_name,
            r.name AS region_name,
            lp.payment_amount,
            lp.payment_method,
            ui.invoice_number,
            lp.notes,
            u.full_name AS created_by_name
        FROM loan_payment lp
        LEFT JOIN customer c ON lp.customer_id = c.id
        LEFT JOIN province p ON lp.province_id = p.id
        LEFT JOIN region r ON p.region_id = r.id
        LEFT JOIN unpaid_invoices ui ON lp.invoice_id = ui.id
        LEFT JOIN user u ON lp.created_by = u.id
        {$where_clause}
        ORDER BY lp.payment_date DESC, lp.id DESC
        LIMIT 10000
    ";
    
    $stmtExport = $conn->prepare($export_query);
    if ($stmtExport) {
        if (!empty($params)) {
            $stmtExport->bind_param($param_types, ...$params);
        }
        $stmtExport->execute();
        $export_data = $stmtExport->get_result()->fetch_all(MYSQLI_ASSOC);
        $stmtExport->close();
        
        $filename = 'loan_repayment_history_' . date('Y-m-d_His') . '.csv';
        export_to_csv($export_data, $filename, [
            'id' => 'ID',
            'payment_date' => 'Date Paiement',
            'customer_name' => 'Client',
            'customer_phone' => 'Téléphone',
            'province_name' => 'Dépôt',
            'region_name' => 'Région',
            'payment_amount' => 'Montant (BIF)',
            'payment_method' => 'Méthode',
            'invoice_number' => 'Commande',
            'notes' => 'Notes',
            'created_by_name' => 'Créé par'
        ]);
        exit;
    }
}

// Build WHERE clause for main query
$where_conditions = [];
$params = [];
$param_types = '';

if ($filter_customer_id > 0) {
    $where_conditions[] = "lp.customer_id = ?";
    $params[] = $filter_customer_id;
    $param_types .= 'i';
}

if ($filter_date_from) {
    $where_conditions[] = "DATE(lp.payment_date) >= ?";
    $params[] = $filter_date_from;
    $param_types .= 's';
}

if ($filter_date_to) {
    $where_conditions[] = "DATE(lp.payment_date) <= ?";
    $params[] = $filter_date_to;
    $param_types .= 's';
}

if ($search_term) {
    $where_conditions[] = "(c.full_name LIKE ? OR ui.invoice_number LIKE ? OR lp.notes LIKE ?)";
    $searchPattern = '%' . $search_term . '%';
    $params[] = $searchPattern;
    $params[] = $searchPattern;
    $params[] = $searchPattern;
    $param_types .= 'sss';
}

if ($province_id && !$is_admin) {
    $where_conditions[] = "lp.province_id = ?";
    $params[] = $province_id;
    $param_types .= 'i';
}

$where_clause = !empty($where_conditions) ? 'WHERE ' . implode(' AND ', $where_conditions) : '';

// Fetch loan repayments
$repayments_query = "
    SELECT 
        lp.id,
        lp.payment_date,
        lp.payment_amount,
        lp.payment_method,
        lp.notes,
        c.id AS customer_id,
        c.full_name AS customer_name,
        c.tel AS customer_phone,
        p.name AS province_name,
        r.name AS region_name,
        ui.invoice_number,
        u.full_name AS created_by_name
    FROM loan_payment lp
    LEFT JOIN customer c ON lp.customer_id = c.id
    LEFT JOIN province p ON lp.province_id = p.id
    LEFT JOIN region r ON p.region_id = r.id
    LEFT JOIN unpaid_invoices ui ON lp.invoice_id = ui.id
    LEFT JOIN user u ON lp.created_by = u.id
    {$where_clause}
    ORDER BY lp.payment_date DESC, lp.id DESC
    LIMIT 1000
";

$repayments = [];
$stmtRepayments = $conn->prepare($repayments_query);
if ($stmtRepayments) {
    if (!empty($params)) {
        $stmtRepayments->bind_param($param_types, ...$params);
    }
    $stmtRepayments->execute();
    $repayments = $stmtRepayments->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmtRepayments->close();
}

// Calculate totals
$total_amount = 0;
$total_count = count($repayments);
foreach ($repayments as $repayment) {
    $total_amount += (int)($repayment['payment_amount'] ?? 0);
}

// Fetch customers for filter dropdown
$customers = [];
$stmtCustomers = $conn->prepare("SELECT DISTINCT c.id, c.full_name FROM customer c INNER JOIN loan_payment lp ON c.id = lp.customer_id ORDER BY c.full_name");
if ($stmtCustomers) {
    $stmtCustomers->execute();
    $customers = $stmtCustomers->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmtCustomers->close();
}

include __DIR__ . '/../../includes/header.php';
?>

<div style="padding: 20px; background: #F4F0E4; min-height: calc(100vh - 70px);">
    <div style="max-width: 1400px; margin: 0 auto;">
        <h1 style="color: #4B2F1F; font-size: 32px; margin-bottom: 20px; font-weight: 700;">Historique des Remboursements de Crédit</h1>
        
        <!-- Filters -->
        <div style="background-color: #FFFFFF; padding: 20px; border-radius: 8px; margin-bottom: 20px; box-shadow: 0 2px 4px rgba(0,0,0,0.1);">
            <form method="GET" action="" style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px; align-items: end;">
                <div>
                    <label for="customer_id" style="display: block; color: #4B2F1F; font-weight: 600; margin-bottom: 5px;">Client</label>
                    <select name="customer_id" id="customer_id" style="width: 100%; padding: 10px; border-radius: 8px; border: 1px solid #4B2F1F; font-size: 14px;">
                        <option value="0">Tous les clients</option>
                        <?php foreach ($customers as $customer): ?>
                            <option value="<?php echo $customer['id']; ?>" <?php echo $filter_customer_id === $customer['id'] ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($customer['full_name'], ENT_QUOTES, 'UTF-8'); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div>
                    <label for="date_from" style="display: block; color: #4B2F1F; font-weight: 600; margin-bottom: 5px;">Date Début</label>
                    <input type="date" name="date_from" id="date_from" value="<?php echo htmlspecialchars($filter_date_from, ENT_QUOTES, 'UTF-8'); ?>" style="width: 100%; padding: 10px; border-radius: 8px; border: 1px solid #4B2F1F; font-size: 14px;">
                </div>
                
                <div>
                    <label for="date_to" style="display: block; color: #4B2F1F; font-weight: 600; margin-bottom: 5px;">Date Fin</label>
                    <input type="date" name="date_to" id="date_to" value="<?php echo htmlspecialchars($filter_date_to, ENT_QUOTES, 'UTF-8'); ?>" style="width: 100%; padding: 10px; border-radius: 8px; border: 1px solid #4B2F1F; font-size: 14px;">
                </div>
                
                <div>
                    <label for="search" style="display: block; color: #4B2F1F; font-weight: 600; margin-bottom: 5px;">Recherche</label>
                    <input type="text" name="search" id="search" value="<?php echo htmlspecialchars($search_term, ENT_QUOTES, 'UTF-8'); ?>" placeholder="Client, Commande, Notes..." style="width: 100%; padding: 10px; border-radius: 8px; border: 1px solid #4B2F1F; font-size: 14px;">
                </div>
                
                <div style="display: flex; gap: 10px;">
                    <button type="submit" style="background-color: #4B2F1F; color: #FFFFFF; padding: 10px 20px; border: none; border-radius: 8px; font-size: 14px; font-weight: 600; cursor: pointer; flex: 1;">Filtrer</button>
                    <a href="?" style="background-color: #6c757d; color: #FFFFFF; padding: 10px 20px; border: none; border-radius: 8px; font-size: 14px; font-weight: 600; text-decoration: none; display: inline-block; text-align: center;">Réinitialiser</a>
                </div>
            </form>
            
            <div style="margin-top: 15px; display: flex; justify-content: space-between; align-items: center;">
                <div style="color: #4B2F1F; font-size: 14px;">
                    <strong>Total:</strong> <?php echo number_format($total_count, 0, ',', '.'); ?> remboursement(s) - 
                    <strong>Montant Total:</strong> <?php echo number_format($total_amount, 0, ',', '.'); ?> BIF
                </div>
                <a href="?<?php echo http_build_query(array_merge($_GET, ['export' => 'csv'])); ?>" style="background-color: #28a745; color: #FFFFFF; padding: 10px 20px; border: none; border-radius: 8px; font-size: 14px; font-weight: 600; text-decoration: none; display: inline-block;">Exporter CSV</a>
            </div>
        </div>
        
        <!-- Repayments Table -->
        <div style="background-color: #FFFFFF; padding: 20px; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.1);">
            <?php if (empty($repayments)): ?>
                <p style="text-align: center; color: #666; padding: 40px;">Aucun remboursement trouvé.</p>
            <?php else: ?>
                <div style="overflow-x: auto;">
                    <table style="width: 100%; border-collapse: collapse;">
                        <thead>
                            <tr style="background-color: #4B2F1F; color: #FFFFFF;">
                                <th style="padding: 12px; text-align: left; font-weight: 600;">Date</th>
                                <th style="padding: 12px; text-align: left; font-weight: 600;">Client</th>
                                <th style="padding: 12px; text-align: left; font-weight: 600;">Dépôt</th>
                                <th style="padding: 12px; text-align: left; font-weight: 600;">Région</th>
                                <th style="padding: 12px; text-align: right; font-weight: 600;">Montant (BIF)</th>
                                <th style="padding: 12px; text-align: left; font-weight: 600;">Méthode</th>
                                <th style="padding: 12px; text-align: left; font-weight: 600;">Commande</th>
                                <th style="padding: 12px; text-align: left; font-weight: 600;">Créé par</th>
                                <th style="padding: 12px; text-align: left; font-weight: 600;">Notes</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($repayments as $repayment): ?>
                                <tr style="border-bottom: 1px solid #dee2e6;">
                                    <td style="padding: 12px;">
                                        <?php echo htmlspecialchars(date('Y-m-d H:i', strtotime($repayment['payment_date'])), ENT_QUOTES, 'UTF-8'); ?>
                                    </td>
                                    <td style="padding: 12px;">
                                        <strong><?php echo htmlspecialchars($repayment['customer_name'] ?? 'N/A', ENT_QUOTES, 'UTF-8'); ?></strong>
                                        <?php if (!empty($repayment['customer_phone'])): ?>
                                            <br><small style="color: #666;"><?php echo htmlspecialchars($repayment['customer_phone'], ENT_QUOTES, 'UTF-8'); ?></small>
                                        <?php endif; ?>
                                    </td>
                                    <td style="padding: 12px;">
                                        <?php echo htmlspecialchars($repayment['province_name'] ?? 'N/A', ENT_QUOTES, 'UTF-8'); ?>
                                    </td>
                                    <td style="padding: 12px;">
                                        <?php echo htmlspecialchars($repayment['region_name'] ?? 'N/A', ENT_QUOTES, 'UTF-8'); ?>
                                    </td>
                                    <td style="padding: 12px; text-align: right; font-weight: 600; color: #28a745;">
                                        <?php echo number_format((int)($repayment['payment_amount'] ?? 0), 0, ',', '.'); ?>
                                    </td>
                                    <td style="padding: 12px;">
                                        <?php
                                        $method = str_replace('_', ' ', (string)($repayment['payment_method'] ?? 'cash'));
                                        $method_fr = ($method === 'cash') ? 'Espèces' : (($method === 'bank transfer') ? 'Virement Bancaire' : (($method === 'mobile') ? 'Paiement Mobile' : $method));
                                        echo htmlspecialchars($method_fr, ENT_QUOTES, 'UTF-8');
                                        ?>
                                    </td>
                                    <td style="padding: 12px;">
                                        <?php echo htmlspecialchars($repayment['invoice_number'] ?? 'Remboursement général', ENT_QUOTES, 'UTF-8'); ?>
                                    </td>
                                    <td style="padding: 12px;">
                                        <?php echo htmlspecialchars($repayment['created_by_name'] ?? 'N/A', ENT_QUOTES, 'UTF-8'); ?>
                                    </td>
                                    <td style="padding: 12px;">
                                        <?php echo htmlspecialchars($repayment['notes'] ?? '-', ENT_QUOTES, 'UTF-8'); ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<?php include __DIR__ . '/../../includes/footer.php'; ?>
<?php $conn->close(); ?>







