<?php
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../includes/functions.php';
require_login();

header('Content-Type: application/json');

$conn = get_db_connection();
if (!$conn) {
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => 'Database connection failed: ' . mysqli_connect_error()]);
    exit;
}

try {
    $invoice_id = intval($_GET['id'] ?? 0);
    if ($invoice_id <= 0) {
        http_response_code(400);
        echo json_encode(['success' => false, 'error' => 'Invalid invoice ID']);
        exit;
    }

    // Fetch invoice header
    $stmt = $conn->prepare("
        SELECT i.*, pr.name AS province_name
        FROM invoice i
        LEFT JOIN province pr ON i.province_id = pr.id
        WHERE i.id = ?
    ");
    if (!$stmt) {
        throw new Exception("Prepare failed: " . $conn->error);
    }
    $stmt->bind_param("i", $invoice_id);
    $stmt->execute();
    $invoice = $stmt->get_result()->fetch_assoc();
    $stmt->close();

    if (!$invoice) {
        http_response_code(404);
        echo json_encode(['success' => false, 'error' => 'Invoice not found']);
        exit;
    }

    // Fetch all items for this invoice
    $items = [];
    $stmt = $conn->prepare("
        SELECT ii.quantity, p.name AS product_name, ii.unit_price, p.price_per_crate
        FROM invoice_items ii
        LEFT JOIN product p ON ii.product_id = p.id
        WHERE ii.invoice_id = ?
    ");
    if (!$stmt) {
        throw new Exception("Prepare failed: " . $conn->error);
    }
    $stmt->bind_param("i", $invoice_id);
    $stmt->execute();
    $items = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();

    $response = [
        'success' => true,
        'customer_name' => $invoice['customer_name'] ?? 'N/A',
        'phone_number' => $invoice['phone_number'] ?? 'N/A',
        'paid_amount' => $invoice['paid_amount'] ?? 'N/A',
        'province_name' => $invoice['province_name'] ?? 'N/A',
        'invoice_number' => $invoice['invoice_number'] ?? 'N/A',
        'paid_status' => $invoice['paid_status'] ?? 'N/A',
        'serve_status' => $invoice['serve_status'] ?? 'N/A',
        'created_at' => $invoice['created_at'] ?? 'N/A'
    ];
    if (count($items) > 0) {
        $response['items'] = $items;
    }

    echo json_encode($response);
} catch (Exception $e) {
    http_response_code(500);
    $error_msg = "Error in get_invoice_details.php for invoice_id=$invoice_id: " . $e->getMessage();
    error_log($error_msg);
    echo json_encode(['success' => false, 'error' => $e->getMessage()]);
} finally {
    if ($conn) $conn->close();
}
?>
