<?php
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../includes/functions.php';

require_login();

$allowed_roles = ['Admin', 'DG', 'DGA', 'Board', 'Accountant', 'Operation Manager', 'Stock Manager', 'Caissier_Comptable'];
$role_name = $_SESSION['role_name'] ?? '';
if (!in_array($role_name, $allowed_roles, true)) {
    header("Location: /masunzu_bar_hotel/dashboards/unauthorized.php");
    exit;
}

$conn = get_db_connection();
if (!$conn) {
    http_response_code(500);
    echo "<p style='color: #FF0000; text-align:center; padding:20px;'>Connexion base de données indisponible.</p>";
    include __DIR__ . '/../../includes/footer.php';
    exit;
}

$user_id = (int)($_SESSION['user_id'] ?? 0);
$user_province_id = $_SESSION['province_id'] ?? null;
$user_region_id = null;

if ($user_province_id) {
    $stmt = $conn->prepare("SELECT region_id FROM province WHERE id = ?");
    if ($stmt) {
        $stmt->bind_param("i", $user_province_id);
        $stmt->execute();
        $row = $stmt->get_result()->fetch_assoc();
        $user_region_id = $row['region_id'] ?? null;
        $stmt->close();
    }
}

// Determine province restriction by role
$allowed_provinces = [];
if (in_array($role_name, ['Admin', 'DG', 'DGA', 'Board', 'Accountant'], true)) {
    // No restriction
} elseif ($role_name === 'Operation Manager' && $user_region_id) {
    $stmt = $conn->prepare("SELECT id FROM province WHERE region_id = ?");
    if ($stmt) {
        $stmt->bind_param("i", $user_region_id);
        $stmt->execute();
        $res = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
        foreach ($res as $r) {
            $allowed_provinces[] = (int)$r['id'];
        }
        $stmt->close();
    }
} elseif (in_array($role_name, ['Stock Manager', 'Caissier_Comptable'], true) && $user_province_id) {
    $allowed_provinces[] = (int)$user_province_id;
}

$province_placeholders = '';
$params = [];
if (!empty($allowed_provinces)) {
    $province_placeholders = implode(',', array_fill(0, count($allowed_provinces), '?'));
}

$sql = "
    SELECT 
        pr.name AS province_name,
        pr.id AS province_id,
        ps.product_id,
        ps.quantity,
        ps.reserved_quantity,
        ps.unit_type,
        p.name AS product_name,
        p.crate_quantity,
        p.low_stock_alert_crates,
        p.low_stock_alert_units
    FROM province_stock ps
    JOIN product p ON ps.product_id = p.id
    JOIN province pr ON ps.province_id = pr.id
    WHERE ps.quantity IS NOT NULL
";
if (!empty($province_placeholders)) {
    $sql .= " AND ps.province_id IN ($province_placeholders)";
}
$stmt = $conn->prepare($sql);
if ($stmt) {
    if (!empty($province_placeholders)) {
        $types = str_repeat('i', count($allowed_provinces));
        $stmt->bind_param($types, ...$allowed_provinces);
    }
    $stmt->execute();
    $rows = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();
} else {
    $rows = [];
}

$alerts = [];
$flattened = [];
foreach ($rows as $stock) {
    $crateQuantity = max(1, (int)($stock['crate_quantity'] ?? 12));
    $rawQuantity = (int)($stock['quantity'] ?? 0);
    $reserved = (int)($stock['reserved_quantity'] ?? 0);
    $available = max(0, $rawQuantity - $reserved);

    $availableUnits = $stock['unit_type'] === 'crates'
        ? $available * $crateQuantity
        : $available;
    $availableCrates = $availableUnits / $crateQuantity;

    $thresholdUnits = (int)($stock['low_stock_alert_units'] ?? 0);
    if ($thresholdUnits <= 0) {
        $thresholdCrates = (int)($stock['low_stock_alert_crates'] ?? 0);
        if ($thresholdCrates > 0) {
            $thresholdUnits = $thresholdCrates * $crateQuantity;
        }
    }
    if ($thresholdUnits <= 0) {
        continue; // no alert configured
    }

    if ($availableUnits <= $thresholdUnits) {
        $thresholdCrates = $thresholdUnits / $crateQuantity;
        $provinceName = $stock['province_name'] ?? 'N/A';
        $alertRow = [
            'province_name' => $provinceName,
            'product_name' => $stock['product_name'],
            'available_crates' => (int)$availableCrates,
            'threshold_crates' => (int)$thresholdCrates,
        ];
        $alerts[$provinceName][] = $alertRow;
        $flattened[] = $alertRow;
    }
}

$export = $_GET['export'] ?? '';
if ($export === 'csv' && !empty($flattened)) {
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename=low_stock_alerts_' . date('Ymd_His') . '.csv');
    $out = fopen('php://output', 'w');
    fputcsv($out, ['Province', 'Produit', 'Disponible (caisses)', 'Seuil (caisses)']);
    foreach ($flattened as $row) {
        fputcsv($out, [
            $row['province_name'],
            $row['product_name'],
            $row['available_crates'],
            $row['threshold_crates']
        ]);
    }
    fclose($out);
    exit;
}

if ($export === 'pdf' && !empty($flattened)) {
    header('Content-Type: application/pdf');
    header('Content-Disposition: attachment; filename=low_stock_alerts_' . date('Ymd_His') . '.pdf');

    $rowsForPdf = [];
    $rowsForPdf[] = ['Province', 'Produit', 'Disponible (caisses)', 'Seuil (caisses)'];
    foreach ($flattened as $r) {
        $rowsForPdf[] = [
            $r['province_name'],
            $r['product_name'],
            (int)$r['available_crates'],
            (int)$r['threshold_crates']
        ];
    }

    // Build PDF text stream with column positioning
    $startX = 40;
    $startY = 780;
    $leading = 16;
    $cols = [0, 180, 390, 520]; // x offsets for each column
    $lines = [];

    foreach ($rowsForPdf as $row) {
        $lineCmds = [];
        $lineCmds[] = sprintf("%d %d Td", $startX + $cols[0], $startY);
        $lineCmds[] = "(" . str_replace(['\\', '(', ')'], ['\\\\', '\\(', '\\)'], $row[0]) . ") Tj";
        $lineCmds[] = sprintf("%d 0 Td", $cols[1] - $cols[0]);
        $lineCmds[] = "(" . str_replace(['\\', '(', ')'], ['\\\\', '\\(', '\\)'], $row[1]) . ") Tj";
        $lineCmds[] = sprintf("%d 0 Td", $cols[2] - $cols[1]);
        $lineCmds[] = "(" . $row[2] . ") Tj";
        $lineCmds[] = sprintf("%d 0 Td", $cols[3] - $cols[2]);
        $lineCmds[] = "(" . $row[3] . ") Tj";
        // move to next line: reset X to start, move down by leading
        $lineCmds[] = sprintf("-%d -%d Td", $cols[3] + $startX, $leading);
        $lines[] = implode("\n", $lineCmds);
        // after first line, subsequent lines start from current baseline (already adjusted)
        $startY = 0;
    }

    $stream = "BT /F1 12 Tf\n" . implode("\n", $lines) . "\nET";

    $pdf = "%PDF-1.4\n";
    $offsets = [];
    $objects = [];
    $objects[] = "1 0 obj << /Type /Catalog /Pages 2 0 R >> endobj\n";
    $objects[] = "2 0 obj << /Type /Pages /Kids [3 0 R] /Count 1 >> endobj\n";
    $objects[] = "3 0 obj << /Type /Page /Parent 2 0 R /MediaBox [0 0 612 792] /Contents 4 0 R /Resources << /Font << /F1 5 0 R >> >> >> endobj\n";
    $objects[] = "4 0 obj << /Length " . strlen($stream) . " >> stream\n$stream\nendstream endobj\n";
    $objects[] = "5 0 obj << /Type /Font /Subtype /Type1 /BaseFont /Helvetica >> endobj\n";

    foreach ($objects as $obj) {
        $offsets[] = strlen($pdf);
        $pdf .= $obj;
    }

    $xrefPosition = strlen($pdf);
    $pdf .= "xref\n0 " . (count($offsets) + 1) . "\n";
    $pdf .= "0000000000 65535 f \n";
    foreach ($offsets as $off) {
        $pdf .= sprintf("%010d 00000 n \n", $off);
    }
    $pdf .= "trailer << /Size " . (count($offsets) + 1) . " /Root 1 0 R >>\n";
    $pdf .= "startxref\n" . $xrefPosition . "\n%%EOF";
    echo $pdf;
    exit;
}

include __DIR__ . '/../../includes/header.php';
?>
<div style="padding:20px;">
    <h2 style="color:#4B2F1F;">Alertes Stock Faible</h2>
    <p style="color:#666;">Vue dédiée des alertes stock faible selon vos permissions.</p>
    <?php if (!empty($flattened)): ?>
        <div style="margin: 10px 0 20px; display:flex; gap:10px; flex-wrap:wrap;">
            <a href="?export=csv" style="background:#4B2F1F; color:#F4F0E4; padding:8px 12px; border-radius:6px; text-decoration:none; font-weight:600;">Exporter CSV</a>
            <a href="?export=pdf" style="background:#B71C1C; color:#FFFFFF; padding:8px 12px; border-radius:6px; text-decoration:none; font-weight:600;">Exporter PDF</a>
        </div>
    <?php endif; ?>
    <?php if (empty($alerts)): ?>
        <p style="color:#000;">Aucune alerte de stock faible.</p>
    <?php else: ?>
        <?php foreach ($alerts as $provinceName => $items): ?>
            <div style="background:#FFFFFF; border-radius:10px; padding:15px; margin-bottom:15px; box-shadow:0 3px 10px rgba(0,0,0,0.1);">
                <h4 style="color:#4B2F1F; margin:0 0 10px 0;"><?php echo htmlspecialchars($provinceName, ENT_QUOTES, 'UTF-8'); ?></h4>
                <table style="width:100%; border-collapse:collapse;">
                    <thead>
                        <tr style="background:#f4f0e4; color:#4B2F1F;">
                            <th style="padding:8px; text-align:left;">Produit</th>
                            <th style="padding:8px; text-align:right;">Disponible (caisses)</th>
                            <th style="padding:8px; text-align:right;">Seuil (caisses)</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($items as $item): ?>
                        <tr>
                            <td style="padding:8px; border-bottom:1px solid #eee;"><?php echo htmlspecialchars($item['product_name'], ENT_QUOTES, 'UTF-8'); ?></td>
                            <td style="padding:8px; text-align:right; border-bottom:1px solid #eee;"><?php echo number_format((int)$item['available_crates'], 0, ',', '.'); ?></td>
                            <td style="padding:8px; text-align:right; border-bottom:1px solid #eee;"><?php echo number_format((int)$item['threshold_crates'], 0, ',', '.'); ?></td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php endforeach; ?>
    <?php endif; ?>
</div>
<?php include __DIR__ . '/../../includes/footer.php'; ?>
