<?php
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../includes/functions.php';
require_once __DIR__ . '/../../libs/tcpdf/tcpdf.php';
require_login();

// Restrict to Caissier_Comptable role
if ($_SESSION['role_name'] !== 'Caissier_Comptable') {
    error_log("Unauthorized access to balance_management.php by user {$_SESSION['full_name']} ({$_SESSION['role_name']})");
    header("Location: /masunzu_bar_hotel/dashboards/operation_dashboard.php");
    exit;
}

// Initialize database connection
$conn = get_db_connection();
if (!$conn) {
    error_log("Database connection failed in balance_management.php: " . mysqli_connect_error());
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Échec de la connexion à la base de données. Contactez l'administrateur.</p>";
    include __DIR__ . '/../../includes/footer.php';
    exit;
}

// Fetch province_id if not set
if (!isset($_SESSION['province_id'])) {
    $stmt = $conn->prepare("SELECT province_id FROM user WHERE id = ?");
    $stmt->bind_param("i", $_SESSION['user_id']);
    $stmt->execute();
    $result = $stmt->get_result();
    $user = $result->fetch_assoc();
    $_SESSION['province_id'] = $user['province_id'] ?? null;
    $stmt->close();
}
$province_id = $_SESSION['province_id'];

if (!$province_id) {
    error_log("No province_id for user {$_SESSION['full_name']} (ID: {$_SESSION['user_id']})");
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Erreur: Aucune province assignée. Contactez l'administrateur.</p>";
    include __DIR__ . '/../../includes/footer.php';
    exit;
}

// Fetch province name for display purposes
$province_name = 'Inconnu';
$stmt = $conn->prepare("SELECT name FROM province WHERE id = ?");
if ($stmt) {
    $stmt->bind_param("i", $province_id);
    $stmt->execute();
    $province_name = $stmt->get_result()->fetch_assoc()['name'] ?? 'Inconnu';
    $stmt->close();
} else {
    error_log("Failed to prepare province name lookup in balance_management.php: " . $conn->error);
}

// Check active shift
$active_shift = null;
$stmt = $conn->prepare("SELECT id, shift_start, beginning_balance, deposits FROM cashier_balance WHERE cashier_id = ? AND shift_end IS NULL ORDER BY shift_start DESC LIMIT 1");
if (!$stmt) {
    error_log("Query preparation failed for active shift: " . $conn->error);
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Erreur de base de données. Contactez l'administrateur.</p>";
    include __DIR__ . '/../../includes/footer.php';
    exit;
}
$stmt->bind_param("i", $_SESSION['user_id']);
$stmt->execute();
$result = $stmt->get_result();
if ($result && $result->num_rows > 0) {
    $active_shift = $result->fetch_assoc();
}
$stmt->close();

$shift_start = $active_shift['shift_start'] ?? null;
$shift_deposits = $active_shift ? (int)($active_shift['deposits'] ?? 0) : 0;
$shift_sales_deposits = 0;
$shift_loan_repayments = 0;
$shift_expenses = 0;
$shift_transfers = 0;
$current_balance = $active_shift ? (int)$active_shift['beginning_balance'] + $shift_deposits : 0;

if ($active_shift && $shift_start) {
    // Calculate sales deposits (from paid_invoices and order table where payment_method != 'loan')
    $stmt = $conn->prepare("
        SELECT COALESCE(SUM(amount), 0) AS total_sales 
        FROM (
            SELECT paid_amount AS amount FROM paid_invoices 
            WHERE created_by = ? AND paid_at >= ? AND payment_method != 'loan'
            UNION ALL
            SELECT paid_amount AS amount FROM `order` 
            WHERE created_by = ? AND created_at >= ? AND paid_amount IS NOT NULL AND paid_amount > 0
        ) AS sales
    ");
    if ($stmt) {
        $stmt->bind_param("isis", $_SESSION['user_id'], $shift_start, $_SESSION['user_id'], $shift_start);
        $stmt->execute();
        $shift_sales_deposits = (int)($stmt->get_result()->fetch_assoc()['total_sales'] ?? 0);
        $stmt->close();
    }

    // Calculate loan repayments (from loan_payment table)
    $stmt = $conn->prepare("SELECT COALESCE(SUM(payment_amount), 0) AS total_loan_repayments FROM loan_payment WHERE created_by = ? AND payment_date >= ?");
    if ($stmt) {
        $stmt->bind_param("is", $_SESSION['user_id'], $shift_start);
        $stmt->execute();
        $shift_loan_repayments = (int)($stmt->get_result()->fetch_assoc()['total_loan_repayments'] ?? 0);
        $stmt->close();
    }

    $stmt = $conn->prepare("SELECT COALESCE(SUM(amount), 0) AS total_expenses FROM expense WHERE cashier_id = ? AND created_at >= ?");
    if (!$stmt) {
        error_log("Query preparation failed for shift expenses snapshot: " . $conn->error);
    } else {
        $stmt->bind_param("is", $_SESSION['user_id'], $shift_start);
        $stmt->execute();
        $shift_expenses = (int)($stmt->get_result()->fetch_assoc()['total_expenses'] ?? 0);
        $stmt->close();
    }

    $stmt = $conn->prepare("SELECT COALESCE(SUM(amount), 0) AS total_transferred FROM cash_transfer WHERE cashier_id = ? AND transfer_date >= ?");
    if (!$stmt) {
        error_log("Query preparation failed for shift transfers snapshot: " . $conn->error);
    } else {
        $stmt->bind_param("is", $_SESSION['user_id'], $shift_start);
        $stmt->execute();
        $shift_transfers = (int)($stmt->get_result()->fetch_assoc()['total_transferred'] ?? 0);
        $stmt->close();
    }

    $current_balance = (int)$active_shift['beginning_balance'] + $shift_deposits - $shift_expenses - $shift_transfers;
    if ($current_balance < 0) {
        $current_balance = 0;
    }
}

// Handle form submission
$error = '';
$success = '';
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['start_shift'])) {
        if ($active_shift) {
            $error = 'Une session active est déjà en cours.';
        } else {
            $beginning_balance = isset($_POST['beginning_balance']) ? (int)round((float)$_POST['beginning_balance']) : 0;
            $notes = trim($_POST['notes'] ?? '');
            $start_cash_total = isset($_POST['start_cash_total']) ? (int)$_POST['start_cash_total'] : null;
            
            if ($beginning_balance < 0) {
                $error = 'Le solde initial ne peut pas être négatif.';
            } else {
                // If beginning balance > 0, require cash count confirmation
                if ($beginning_balance > 0) {
                    if ($start_cash_total === null || $start_cash_total < 0) {
                        $error = 'Veuillez confirmer le montant des espèces comptées pour le solde initial.';
                    } else if ($start_cash_total !== $beginning_balance) {
                        $error = 'Le montant compté doit correspondre au solde initial saisi.';
                    }
                }

                if (!$error) {
                    // Ensure cashier_account table exists
                    $createCashierAccountSql = "
                        CREATE TABLE IF NOT EXISTS cashier_account (
                            id INT AUTO_INCREMENT PRIMARY KEY,
                            cashier_id INT NOT NULL,
                            shift_id INT NULL,
                            transaction_type ENUM('SHIFT_START','SHIFT_END','INVOICE_PAYMENT','LOAN_REPAYMENT','REQUISITION_PAYMENT','EXPENSE','BANK_DEPOSIT','CASH_TRANSFER','ADJUSTMENT') NOT NULL,
                            direction ENUM('CREDIT','DEBIT') NOT NULL,
                            amount BIGINT(20) NOT NULL,
                            reference_type ENUM('SHIFT','INVOICE','LOAN_PAYMENT','REQUISITION','EXPENSE','BANK_DEPOSIT','CASH_TRANSFER','MANUAL') NOT NULL,
                            reference_id INT NULL,
                            description VARCHAR(255) DEFAULT NULL,
                            created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
                            created_by INT NULL,
                            INDEX idx_cashier_created (cashier_id, created_at),
                            INDEX idx_shift (shift_id),
                            INDEX idx_reference (reference_type, reference_id),
                            CONSTRAINT fk_ca_cashier FOREIGN KEY (cashier_id) REFERENCES user(id) ON DELETE CASCADE,
                            CONSTRAINT fk_ca_shift FOREIGN KEY (shift_id) REFERENCES cashier_balance(id) ON DELETE SET NULL,
                            CONSTRAINT fk_ca_creator FOREIGN KEY (created_by) REFERENCES user(id) ON DELETE SET NULL
                        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci
                    ";
                    @$conn->query($createCashierAccountSql);
                    
                    $stmt = $conn->prepare("INSERT INTO cashier_balance (cashier_id, shift_start, beginning_balance, notes, created_at) VALUES (?, NOW(), ?, ?, NOW())");
                    if (!$stmt) {
                        $error = 'Échec de la préparation de la création de la session. ' . $conn->error;
                    } else {
                        $stmt->bind_param("ids", $_SESSION['user_id'], $beginning_balance, $notes);
                        if ($stmt->execute()) {
                            $shift_id = (int)$conn->insert_id;
                            $stmt->close();
                            
                            // Enregistrer dans cashier_account si solde initial > 0
                            if ($beginning_balance > 0) {
                                $stmtAccount = $conn->prepare("
                                    INSERT INTO cashier_account (cashier_id, shift_id, transaction_type, direction, amount, reference_type, reference_id, created_by, description)
                                    VALUES (?, ?, 'SHIFT_START', 'CREDIT', ?, 'SHIFT', ?, ?, ?)
                                ");
                                if ($stmtAccount) {
                                    $description = "Début de session - Solde initial";
                                    $stmtAccount->bind_param("iiiiis", $_SESSION['user_id'], $shift_id, $beginning_balance, $shift_id, $_SESSION['user_id'], $description);
                                    if (!$stmtAccount->execute()) {
                                        error_log("Failed to record SHIFT_START in cashier_account: " . $stmtAccount->error);
                                    }
                                    $stmtAccount->close();
                                }
                            }
                            
                            $logMessage = "Caissier a démarré la session avec solde {$beginning_balance} BIF";
                            if ($start_cash_total !== null && $start_cash_total > 0) {
                                $logMessage .= " | Espèces comptées: {$start_cash_total} BIF";
                            }
                            log_action($_SESSION['user_id'], 'start_shift', $logMessage);
                            header("Location: /masunzu_bar_hotel/modules/cashiers/balance_management.php");
                            exit;
                        } else {
                            $error = 'Échec du démarrage de la session: ' . $stmt->error;
                            error_log("Shift start failed: " . $stmt->error);
                            $stmt->close();
                        }
                    }
                }
            }
        }
    } elseif (isset($_POST['end_shift']) && $active_shift) {
        $notes = trim($_POST['notes'] ?? '');
        $cash_count_total = isset($_POST['cash_count_total']) ? (int)$_POST['cash_count_total'] : null;
        $cash_difference_post = isset($_POST['cash_difference']) ? (int)$_POST['cash_difference'] : null;
        
        // Calculate deposits (sales + loan repayments) for this shift
        $shift_start = $active_shift['shift_start'];
        $sales_deposits = 0;
        $loan_repayments = 0;
        
        // Calculate sales deposits (from paid_invoices and order table where payment_method != 'loan')
        $stmt = $conn->prepare("
            SELECT COALESCE(SUM(amount), 0) AS total_sales 
            FROM (
                SELECT paid_amount AS amount FROM paid_invoices 
                WHERE created_by = ? AND paid_at >= ? AND payment_method != 'loan'
                UNION ALL
                SELECT paid_amount AS amount FROM `order` 
                WHERE created_by = ? AND created_at >= ? AND paid_amount IS NOT NULL AND paid_amount > 0
            ) AS sales
        ");
        if ($stmt) {
            $stmt->bind_param("isis", $_SESSION['user_id'], $shift_start, $_SESSION['user_id'], $shift_start);
            $stmt->execute();
            $sales_deposits = (int)($stmt->get_result()->fetch_assoc()['total_sales'] ?? 0);
            $stmt->close();
        }
        
        // Calculate loan repayments (from loan_payment table)
        $stmt = $conn->prepare("SELECT COALESCE(SUM(payment_amount), 0) AS total_loan_repayments FROM loan_payment WHERE created_by = ? AND payment_date >= ?");
        if ($stmt) {
            $stmt->bind_param("is", $_SESSION['user_id'], $shift_start);
            $stmt->execute();
            $loan_repayments = (int)($stmt->get_result()->fetch_assoc()['total_loan_repayments'] ?? 0);
            $stmt->close();
        }
        
        $deposits = $sales_deposits + $loan_repayments;

        $stmt = $conn->prepare("SELECT SUM(amount) as total_withdrawals FROM expense WHERE cashier_id = ? AND created_at >= ?");
        if (!$stmt) {
            error_log("Query preparation failed for total withdrawals: " . $conn->error);
            echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Erreur de base de données. Contactez l'administrateur.</p>";
            include __DIR__ . '/../../includes/footer.php';
            exit;
        }
        $stmt->bind_param("is", $_SESSION['user_id'], $active_shift['shift_start']);
        $stmt->execute();
        $result = $stmt->get_result();
        $withdrawals = $result->fetch_assoc()['total_withdrawals'] ?? 0.00;
        $stmt->close();

        $end_balance = $active_shift['beginning_balance'] + $deposits - $withdrawals;
        $end_balance = (int)$end_balance; // Ensure it's an integer
        
        // Allow cash_count_total to be 0 when end_balance is 0 (no payments made)
        // But still require the cashier to explicitly enter 0 via the denomination breakdown
        if ($cash_count_total === null) {
            $error = 'Veuillez fournir votre comptage d\'espèces en utilisant la répartition. Si vous n\'avez pas d\'espèces, entrez 0 pour toutes les dénominations.';
        } else if ($cash_count_total < 0) {
            $error = 'Le comptage d\'espèces ne peut pas être négatif.';
        }
        
        if (!$error) {
            // Calculate difference
            $count_difference = $cash_count_total - $end_balance;
            
            // Special case: If end_balance is 0, allow cash_count_total to be 0
            // This allows cashiers to close shifts when no payments were made
            if ($end_balance == 0 && $cash_count_total == 0) {
                // Explicitly allow: no payments made, no cash counted - valid scenario
                $count_difference = 0;
                // No error - shift can be closed
            } else if ($end_balance == 0 && $cash_count_total > 0) {
                // End balance is 0 but cashier counted cash - this is an excess
                $error = 'Le comptage d\'espèces dépasse le montant attendu. Aucun excédent n\'est autorisé. Montant attendu: 0 BIF (aucun paiement effectué pendant cette session). Veuillez vérifier votre comptage.';
            } else if ($end_balance > 0) {
                // Normal validation when end_balance > 0
                if ($cash_count_total < $end_balance) {
                    $error = 'Le comptage d\'espèces est inférieur au montant attendu. Veuillez réconcilier avant de fermer la session.';
                } elseif ($count_difference > 0) {
                    $error = 'Le comptage d\'espèces dépasse le montant attendu. Aucun excédent n\'est autorisé. Veuillez vérifier et ajuster.';
                } elseif ($count_difference < 0) {
                    $error = 'Le comptage d\'espèces est inférieur au montant attendu. Veuillez réconcilier avant de fermer la session.';
                }
            } else if ($end_balance < 0) {
                // System issue - negative balance shouldn't happen, but allow closure if count matches
                if ($cash_count_total != $end_balance) {
                    $error = 'Le comptage d\'espèces ne correspond pas au solde attendu. Veuillez réconcilier avant de fermer la session.';
                } else {
                    $count_difference = 0;
                }
            }
        }

        // Generate PDF journal
        $today_start = date('Y-m-d 00:00:00', strtotime($active_shift['shift_start']));
        $today_end = date('Y-m-d 23:59:59');
        $today_query = "
            SELECT o.invoice_number, o.customer_name, o.paid_amount, o.created_at, 'Credit' as type
            FROM `order` o
            WHERE o.created_by = ? AND o.created_at BETWEEN ? AND ?
            UNION ALL
            SELECT e.id as invoice_number, e.receipt_number as customer_name, e.amount as paid_amount, e.created_at, 'Debit' as type
            FROM expense e
            WHERE e.cashier_id = ? AND e.created_at BETWEEN ? AND ?
            ORDER BY created_at";
        $stmt = $conn->prepare($today_query);
        if (!$stmt) {
            error_log("Query preparation failed for PDF transactions: " . $conn->error);
            echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Erreur de base de données. Contactez l'administrateur.</p>";
            include __DIR__ . '/../../includes/footer.php';
            exit;
        }
                $stmt->bind_param("ississ", $_SESSION['user_id'], $today_start, $today_end, $_SESSION['user_id'], $today_start, $today_end);
                $stmt->execute();
                $today_transactions = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
                $stmt->close();

                // Expected amount should never be negative (if balance is negative, expected cash is 0)
                $expected_cash = max(0, (int)$end_balance);
                $count_difference = $cash_count_total - $expected_cash;

                // Create PDF
                $pdf = new TCPDF(PDF_PAGE_ORIENTATION, PDF_UNIT, PDF_PAGE_FORMAT, true, 'UTF-8', false);
                $pdf->SetCreator(PDF_CREATOR);
                $pdf->SetAuthor($_SESSION['full_name']);
                $pdf->SetTitle('Daily Journal - ' . $_SESSION['full_name'] . ' - ' . date('Ymd'));
        $pdf->SetMargins(10, 10, 10);
        $pdf->SetAutoPageBreak(TRUE, 10);
        $pdf->AddPage();
        $pdf->SetFont('helvetica', '', 12);

        // Header
        $pdf->Write(0, 'Masunzu Bar Hotel - Journal Quotidien', '', 0, 'C', true, 0, false, false, 0);
        $pdf->Write(0, 'Caissier: ' . $_SESSION['full_name'], '', 0, 'L', true, 0, false, false, 0);
        $pdf->Write(0, 'Date: ' . date('Y-m-d'), '', 0, 'L', true, 0, false, false, 0);
        $pdf->Ln(10);

        // Starting Balance
        $pdf->Write(0, 'Solde Initial: ' . number_format((int)($active_shift['beginning_balance'] ?? 0), 0, ',', '.') . ' BIF', '', 0, 'L', true, 0, false, false, 0);
        $pdf->Ln(5);

        // Transactions Table
        $pdf->Write(0, 'Transactions:', '', 0, 'L', true, 0, false, false, 0);
        $html = '
        <table border="1" cellpadding="4">
            <tr style="background-color: #4B2F1F; color: #F4F0E4;">
                <th>Date</th>
                <th>Numéro Commande</th>
                <th>Description</th>
                <th>Crédit (BIF)</th>
                <th>Débit (BIF)</th>
            </tr>';
        foreach ($today_transactions as $txn) {
            $bg_color = $txn['type'] === 'Credit' ? '#E6FFE6' : '#FFE6E6';
            $credit = $txn['type'] === 'Credit' ? number_format((int)$txn['paid_amount'], 0, ',', '.') : '';
            $debit = $txn['type'] === 'Debit' ? number_format((int)$txn['paid_amount'], 0, ',', '.') : '';
            $description = $txn['customer_name'];
            $html .= '
            <tr style="background-color: ' . $bg_color . ';">
                <td>' . date('Y-m-d H:i', strtotime($txn['created_at'])) . '</td>
                <td>' . htmlspecialchars($txn['invoice_number'], ENT_QUOTES, 'UTF-8') . '</td>
                <td>' . htmlspecialchars($description, ENT_QUOTES, 'UTF-8') . '</td>
                <td style="text-align: right;">' . $credit . '</td>
                <td style="text-align: right;">' . $debit . '</td>
            </tr>';
        }
        $html .= '</table>';
        $pdf->writeHTML($html, true, false, true, false, '');

        // Totals
        $pdf->Ln(5);
                $pdf->Write(0, 'Total Crédits: ' . number_format($deposits, 0, ',', '.') . ' BIF', '', 0, 'L', true, 0, false, false, 0);
                $pdf->Write(0, 'Total Débits: ' . number_format($withdrawals, 0, ',', '.') . ' BIF', '', 0, 'L', true, 0, false, false, 0);
                $pdf->Write(0, 'Solde Disponible: ' . number_format((int)$end_balance, 0, ',', '.') . ' BIF', '', 0, 'L', true, 0, false, false, 0);
                $pdf->Write(0, 'Espèces Comptées: ' . number_format($cash_count_total, 0, ',', '.') . ' BIF', '', 0, 'L', true, 0, false, false, 0);
                $pdf->Write(0, 'Différence: ' . number_format($count_difference, 0, ',', '.') . ' BIF', '', 0, 'L', true, 0, false, false, 0);

                // Save PDF
                $journal_dir = null;
                $candidate_dirs = [
                    __DIR__ . '/../../storage/journals/',
            rtrim(sys_get_temp_dir(), DIRECTORY_SEPARATOR) . '/masunzu_journals/'
        ];
        foreach ($candidate_dirs as $dir) {
            if (!is_dir($dir)) {
                if (!@mkdir($dir, 0755, true)) {
                    continue;
                }
            }
            if (is_writable($dir)) {
                $journal_dir = $dir;
                break;
            }
        }
        if ($journal_dir === null) {
            error_log('Unable to find or create a writable journals directory.');
            echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Impossible de stocker le PDF du journal. Contactez l'administrateur.</p>";
            include __DIR__ . '/../../includes/footer.php';
            exit;
        }
        $cashier_name = strtolower(str_replace(' ', '_', $_SESSION['full_name']));
        $pdf_filename = $cashier_name . '_' . date('Ymd') . '_journal.pdf';
        $pdf_path = $journal_dir . $pdf_filename;
        $pdf->Output($pdf_path, 'F');

        // Store in receipts table
        $stmt = $conn->prepare("INSERT INTO receipts (receipt_id, transaction_type, transaction_id, receipt_number, customer_name, phone_number, total_amount, created_at, created_by, pdf_content) VALUES (UUID(), 'direct_sale', 0, ?, ?, '', ?, NOW(), ?, NULL)");
        if (!$stmt) {
            error_log("Query preparation failed for receipts insertion: " . $conn->error);
            echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Erreur de base de données. Contactez l'administrateur.</p>";
            include __DIR__ . '/../../includes/footer.php';
            exit;
        }
        $stmt->bind_param("ssdi", $pdf_filename, $_SESSION['full_name'], $end_balance, $_SESSION['user_id']);
        $stmt->execute();
                $stmt->close();

                // Update cashier_balance
                // IMPORTANT: end_balance should be the EXPECTED balance (calculated), not the counted cash
                // The counted cash is stored separately in cashier_shift_closure table
                $stmt = $conn->prepare("UPDATE cashier_balance SET shift_end = NOW(), end_balance = ?, deposits = ?, withdrawals = ?, notes = ? WHERE id = ?");
                if (!$stmt) {
                    error_log("Query preparation failed for cashier_balance update: " . $conn->error);
                    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Erreur de base de données. Contactez l'administrateur.</p>";
                    include __DIR__ . '/../../includes/footer.php';
                    exit;
                }
                $stmt->bind_param("ddisi", $end_balance, $deposits, $withdrawals, $notes, $active_shift['id']);
                if ($stmt->execute()) {
                    // Persist closure details
        $create_closure_table_sql = "
                        CREATE TABLE IF NOT EXISTS cashier_shift_closure (
                            id INT AUTO_INCREMENT PRIMARY KEY,
                            shift_id INT NOT NULL,
                            cashier_id INT NOT NULL,
                            province_id INT DEFAULT NULL,
                            expected_amount BIGINT NOT NULL,
                            counted_amount BIGINT NOT NULL,
                            difference_amount INT NOT NULL,
                            deposited_to_bank ENUM('yes', 'no') NOT NULL DEFAULT 'no',
                            transferred_to_accountant ENUM('yes', 'no') NOT NULL DEFAULT 'no',
                            created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
                            CONSTRAINT fk_closure_shift FOREIGN KEY (shift_id) REFERENCES cashier_balance(id) ON DELETE CASCADE,
                            CONSTRAINT fk_closure_cashier FOREIGN KEY (cashier_id) REFERENCES user(id) ON DELETE CASCADE
                        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci
                    ";
                    if (!$conn->query($create_closure_table_sql)) {
                        error_log("Failed creating cashier_shift_closure table: " . $conn->error);
                    } else {
                        // Migrate existing table: remove denomination columns and add YES/NO columns if they don't exist
                        // Check and drop columns one by one (MySQL doesn't support DROP COLUMN IF EXISTS)
                        $columns_to_drop = [
                            'notes_10000', 'notes_5000', 'notes_2000', 'notes_1000', 'notes_500', 'notes_100',
                            'coins_50', 'coins_20', 'coins_10', 'coins_5', 'coins_1',
                            'deposited_to_bank_amount', 'transferred_to_accountant_amount',
                            'deposited_to_bank_at', 'transferred_to_accountant_at'
                        ];
                        
                        foreach ($columns_to_drop as $col) {
                            $check_col = $conn->query("SHOW COLUMNS FROM cashier_shift_closure LIKE '$col'");
                            if ($check_col && $check_col->num_rows > 0) {
                                @$conn->query("ALTER TABLE cashier_shift_closure DROP COLUMN `$col`");
                            }
                            if ($check_col instanceof mysqli_result) {
                                $check_col->close();
                            }
                        }
                        
                        // Add new columns if they don't exist
                        $check_deposited = $conn->query("SHOW COLUMNS FROM cashier_shift_closure LIKE 'deposited_to_bank'");
                        if (!$check_deposited || $check_deposited->num_rows == 0) {
                            $conn->query("ALTER TABLE cashier_shift_closure ADD COLUMN deposited_to_bank ENUM('yes', 'no') NOT NULL DEFAULT 'no'");
                        }
                        if ($check_deposited) $check_deposited->close();
                        
                        $check_transferred = $conn->query("SHOW COLUMNS FROM cashier_shift_closure LIKE 'transferred_to_accountant'");
                        if (!$check_transferred || $check_transferred->num_rows == 0) {
                            $conn->query("ALTER TABLE cashier_shift_closure ADD COLUMN transferred_to_accountant ENUM('yes', 'no') NOT NULL DEFAULT 'no'");
                        }
                        if ($check_transferred) $check_transferred->close();
                        
                        $stmt_closure = $conn->prepare("
                            INSERT INTO cashier_shift_closure (
                                shift_id, cashier_id, province_id, expected_amount, counted_amount, difference_amount,
                                deposited_to_bank, transferred_to_accountant
                            )
                            VALUES (?, ?, ?, ?, ?, ?, 'no', 'no')
                        ");
                        if ($stmt_closure) {
                            $stmt_closure->bind_param(
                                "iiiiii",
                                $active_shift['id'],
                                $_SESSION['user_id'],
                                $province_id,
                                $expected_cash,
                                $cash_count_total,
                                $count_difference
                            );
                            if (!$stmt_closure->execute()) {
                                error_log("Failed to insert shift closure data: " . $stmt_closure->error);
                            }
                            $stmt_closure->close();
                        } else {
                            error_log("Failed to prepare shift closure insert: " . $conn->error);
                        }
                    }

                    $success = 'Session fermée avec succès! Attendu: ' . number_format($expected_cash, 0, ',', '.') . ' BIF, Compté: ' . number_format($cash_count_total, 0, ',', '.') . ' BIF. Journal sauvegardé: ' . $pdf_filename;
                    log_action($_SESSION['user_id'], 'end_shift', "Caissier a fermé la session. Attendu {$expected_cash} BIF, compté {$cash_count_total} BIF, différence {$count_difference} BIF, journal: $pdf_filename");
                    header("Location: /masunzu_bar_hotel/modules/cashiers/balance_management.php");
                    exit;
                } else {
            $error = 'Échec de la fermeture de la session: ' . $stmt->error;
            error_log("Shift end failed: " . $stmt->error);
        }
        $stmt->close();
    }
}

// Check and update cashier_shift_closure table structure if needed
$check_table_stmt = $conn->query("SHOW TABLES LIKE 'cashier_shift_closure'");
$closure_table_exists = false;
if ($check_table_stmt && $check_table_stmt->num_rows > 0) {
    $closure_table_exists = true;
    
    // Migrate existing table: remove denomination columns if they exist
    $columns_to_drop = [
        'notes_10000', 'notes_5000', 'notes_2000', 'notes_1000', 'notes_500', 'notes_100',
        'coins_50', 'coins_20', 'coins_10', 'coins_5', 'coins_1',
        'deposited_to_bank_amount', 'transferred_to_accountant_amount',
        'deposited_to_bank_at', 'transferred_to_accountant_at'
    ];
    
    foreach ($columns_to_drop as $col) {
        $check_col = $conn->query("SHOW COLUMNS FROM cashier_shift_closure LIKE '$col'");
        if ($check_col && $check_col->num_rows > 0) {
            @$conn->query("ALTER TABLE cashier_shift_closure DROP COLUMN `$col`");
        }
        if ($check_col instanceof mysqli_result) {
            $check_col->close();
        }
    }
    
    // Add new YES/NO columns if they don't exist
    $check_deposited = $conn->query("SHOW COLUMNS FROM cashier_shift_closure LIKE 'deposited_to_bank'");
    if (!$check_deposited || $check_deposited->num_rows == 0) {
        $conn->query("ALTER TABLE cashier_shift_closure ADD COLUMN deposited_to_bank ENUM('yes', 'no') NOT NULL DEFAULT 'no'");
    }
    if ($check_deposited instanceof mysqli_result) {
        $check_deposited->close();
    }
    
    $check_transferred = $conn->query("SHOW COLUMNS FROM cashier_shift_closure LIKE 'transferred_to_accountant'");
    if (!$check_transferred || $check_transferred->num_rows == 0) {
        $conn->query("ALTER TABLE cashier_shift_closure ADD COLUMN transferred_to_accountant ENUM('yes', 'no') NOT NULL DEFAULT 'no'");
    }
    if ($check_transferred instanceof mysqli_result) {
        $check_transferred->close();
    }
}
if ($check_table_stmt instanceof mysqli_result) {
    $check_table_stmt->free();
}

// Fetch recent shifts

if ($closure_table_exists) {
    $recent_shifts_query = "SELECT cb.id,
                                   cb.shift_start,
                                   cb.shift_end,
                                   cb.beginning_balance,
                                   cb.end_balance,
                                   cb.deposits,
                                   cb.withdrawals,
                                   cb.notes,
                                   u.full_name AS cashier_name,
                                   p.name AS province_name,
                                   csc.counted_amount,
                                   csc.difference_amount,
                                   csc.deposited_to_bank,
                                   csc.transferred_to_accountant,
                                   csc.created_at AS closure_recorded_at
                            FROM cashier_balance cb
                            LEFT JOIN cashier_shift_closure csc ON csc.shift_id = cb.id
                            LEFT JOIN user u ON u.id = cb.cashier_id
                            LEFT JOIN province p ON p.id = u.province_id
                            WHERE cb.cashier_id = ?
                            ORDER BY cb.created_at DESC, cb.shift_start DESC
                            LIMIT 5";
} else {
    $recent_shifts_query = "SELECT cb.id,
                                   cb.shift_start,
                                   cb.shift_end,
                                   cb.beginning_balance,
                                   cb.end_balance,
                                   cb.deposits,
                                   cb.withdrawals,
                                   cb.notes,
                                   u.full_name AS cashier_name,
                                   p.name AS province_name
                            FROM cashier_balance cb
                            LEFT JOIN user u ON u.id = cb.cashier_id
                            LEFT JOIN province p ON p.id = u.province_id
                            WHERE cb.cashier_id = ?
                            ORDER BY cb.created_at DESC, cb.shift_start DESC
                            LIMIT 5";
}
$stmt = $conn->prepare($recent_shifts_query);
$recent_shifts = [];
$shift_invoice_payments = [];
$recent_shift_loan_repayments = [];
if (!$stmt) {
    $recent_shifts = [];
} else {
    $stmt->bind_param("i", $_SESSION['user_id']);
    $stmt->execute();
    $recent_shifts = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();
    
    // Calculate invoice payments (excluding loans) and loan repayments for each shift before closing connection
    foreach ($recent_shifts as $shift) {
    $shift_id = (int)$shift['id'];
    $shift_start_time = $shift['shift_start'];
    $shift_end_time = $shift['shift_end'] ?? date('Y-m-d H:i:s');
    $cashier_id_for_shift = $_SESSION['user_id'];
    
    // Query invoice payments excluding loans
    $invoice_stmt = $conn->prepare("
        SELECT COALESCE(SUM(paid_amount), 0) AS total_payments 
        FROM paid_invoices 
        WHERE created_by = ? AND paid_at >= ? AND paid_at <= ? AND payment_method != 'loan'
    ");
    if ($invoice_stmt) {
        $invoice_stmt->bind_param("iss", $cashier_id_for_shift, $shift_start_time, $shift_end_time);
        $invoice_stmt->execute();
        $invoice_result = $invoice_stmt->get_result();
        if ($invoice_result) {
            $invoice_row = $invoice_result->fetch_assoc();
            $shift_invoice_payments[$shift_id] = (int)($invoice_row['total_payments'] ?? 0);
        }
        $invoice_stmt->close();
    } else {
        $shift_invoice_payments[$shift_id] = 0;
    }
    
    // Query loan repayments
    $loan_stmt = $conn->prepare("
        SELECT COALESCE(SUM(payment_amount), 0) AS total_loan_repayments 
        FROM loan_payment 
        WHERE created_by = ? AND payment_date >= ? AND payment_date <= ?
    ");
    if ($loan_stmt) {
        $loan_stmt->bind_param("iss", $cashier_id_for_shift, $shift_start_time, $shift_end_time);
        $loan_stmt->execute();
        $loan_result = $loan_stmt->get_result();
        if ($loan_result) {
            $loan_row = $loan_result->fetch_assoc();
            $recent_shift_loan_repayments[$shift_id] = (int)($loan_row['total_loan_repayments'] ?? 0);
        }
        $loan_stmt->close();
    } else {
        $recent_shift_loan_repayments[$shift_id] = 0;
    }
    }
}

$conn->close();

require_once __DIR__ . '/../../includes/header.php';
?>
<div class="dashboard-container" style="padding: 0; background: linear-gradient(135deg, #4B2F1F 0%, #F4A261 100%); min-height: 100vh;">
    <div class="dashboard-content" style="background-color: #F4F0E4; padding: 20px; border-radius: 0; box-shadow: none; max-width: 100%; margin: 0; width: 100%;">
        <h2 style="color: #4B2F1F; margin-bottom: 10px;">Gestion des Soldes</h2>
        <p style="color: #000000; font-size: 16px;">Gérez les soldes de vos sessions pour Masunzu Bar Hotel.</p>
        <?php if ($error): ?>
            <div style="background-color: #FFFFFF; color: #FF0000; padding: 10px; border-radius: 5px; border: 1px solid #FF0000; margin-bottom: 10px;">
                <?php echo htmlspecialchars($error, ENT_QUOTES, 'UTF-8'); ?>
            </div>
        <?php endif; ?>
        <?php if ($success): ?>
            <div style="background-color: #F4A261; color: #000000; padding: 10px; border-radius: 8px; margin-bottom: 10px;">
                <?php echo htmlspecialchars($success, ENT_QUOTES, 'UTF-8'); ?>
            </div>
        <?php endif; ?>
<?php if ($active_shift): ?>
    <h3 style="color: #4B2F1F; margin-top: 20px; margin-bottom: 10px;">Session Active</h3>
    <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(180px, 1fr)); gap: 14px; background-color: #FFFFFF; border-radius: 10px; padding: 16px; border: 1px solid #E0D6CC; margin-bottom: 10px;">
        <div>
            <div style="color: #4B2F1F; font-size: 12px; font-weight: 600;">Session Démarrée</div>
            <div style="color: #000000; font-size: 16px;"><?php echo date('Y-m-d H:i', strtotime($active_shift['shift_start'])); ?></div>
        </div>
        <div>
            <div style="color: #4B2F1F; font-size: 12px; font-weight: 600;">Solde Initial</div>
            <div style="color: #000000; font-size: 16px;"><?php echo number_format((int)$active_shift['beginning_balance'], 0, ',', '.') . ' BIF'; ?></div>
        </div>
        <div>
            <div style="color: #4B2F1F; font-size: 12px; font-weight: 600;">Dépôts (Ventes)</div>
            <div style="color: #000000; font-size: 16px;"><?php echo number_format($shift_sales_deposits, 0, ',', '.') . ' BIF'; ?></div>
        </div>
        <div>
            <div style="color: #4B2F1F; font-size: 12px; font-weight: 600;">Remboursements Crédits</div>
            <div style="color: #000000; font-size: 16px;"><?php echo number_format($shift_loan_repayments, 0, ',', '.') . ' BIF'; ?></div>
        </div>
        <div>
            <div style="color: #4B2F1F; font-size: 12px; font-weight: 600;">Total Dépôts</div>
            <div style="color: #000000; font-size: 16px;"><?php echo number_format($shift_deposits, 0, ',', '.') . ' BIF'; ?></div>
        </div>
        <div>
            <div style="color: #4B2F1F; font-size: 12px; font-weight: 600;">Dépenses</div>
            <div style="color: #000000; font-size: 16px;"><?php echo number_format($shift_expenses, 0, ',', '.') . ' BIF'; ?></div>
        </div>
        <div>
            <div style="color: #4B2F1F; font-size: 12px; font-weight: 600;">Transferts</div>
            <div style="color: #000000; font-size: 16px;"><?php echo number_format($shift_transfers, 0, ',', '.') . ' BIF'; ?></div>
        </div>
        <div>
            <div style="color: #4B2F1F; font-size: 12px; font-weight: 600;">Solde Actuel</div>
            <div style="color: #2E7D32; font-size: 18px; font-weight: 700;"><?php echo number_format($current_balance, 0, ',', '.') . ' BIF'; ?></div>
        </div>
    </div>
            <form id="end-shift-form" method="POST" style="text-align: left;">
                <input type="hidden" name="cash_count_total" id="cash_count_total" value="">
                <input type="hidden" name="cash_difference" id="cash_difference" value="">
                <input type="hidden" name="notes" id="notes_hidden" value="">
                <button type="button" id="open-cash-count-modal" style="width: 100%; padding: 12px; background-color: #F4A261; color: #000000; border: none; border-radius: 8px; font-size: 16px; cursor: pointer; font-weight: bold;">Fermer Session</button>
            </form>
        <?php else: ?>
            <h3 style="color: #4B2F1F; margin-top: 20px; margin-bottom: 10px;">Démarrer Nouvelle Session</h3>
            <form method="POST" id="start-shift-form" style="text-align: left;">
                <label for="beginning_balance_input" style="color: #4B2F1F; font-size: 16px;">Solde Initial (BIF)</label>
                <input
                    type="number"
                    id="beginning_balance_input"
                    name="beginning_balance"
                    min="0"
                    step="1"
                    style="width: 100%; padding: 12px; margin: 10px 0; border: 2px solid #4B2F1F; border-radius: 8px; font-size: 16px;"
                    value="<?php echo isset($_POST['start_shift']) ? htmlspecialchars($_POST['beginning_balance'] ?? '0', ENT_QUOTES, 'UTF-8') : '0'; ?>"
                    required
                >
                <label for="start-notes" style="color: #4B2F1F; font-size: 16px;">Notes (Optionnel)</label>
                <textarea
                    id="start-notes"
                    name="notes"
                    placeholder="Ajouter des notes sur la session"
                    style="width: 100%; padding: 12px; margin: 10px 0; border: 2px solid #4B2F1F; border-radius: 8px; font-size: 16px;"
                ><?php echo isset($_POST['start_shift']) ? htmlspecialchars($_POST['notes'] ?? '', ENT_QUOTES, 'UTF-8') : ''; ?></textarea>
                <button type="button" id="open-start-cash-modal" style="width: 100%; padding: 12px; background-color: #F4A261; color: #000000; border: none; border-radius: 8px; font-size: 16px; cursor: pointer; font-weight: bold;">Démarrer Session</button>
            </form>
            <div id="start-cash-modal" class="cash-modal" style="display: none;">
                <div class="cash-modal-content" style="background: linear-gradient(135deg, #FFFFFF 0%, #F4F0E4 100%); padding: 30px; border-radius: 16px; width: 100%; max-width: 560px; box-shadow: 0 12px 40px rgba(75, 47, 31, 0.3); border: 2px solid #F4A261;">
                    <div style="text-align: center; margin-bottom: 25px;">
                        <h3 style="color: #4B2F1F; margin: 0 0 8px 0; font-size: 24px; font-weight: 600;">Confirmer Espèces d'Ouverture</h3>
                        <div style="width: 60px; height: 3px; background: linear-gradient(90deg, #4B2F1F 0%, #F4A261 100%); margin: 0 auto; border-radius: 2px;"></div>
                    </div>
                    <div style="background-color: rgba(75, 47, 31, 0.05); padding: 20px; border-radius: 12px; margin-bottom: 25px; border-left: 4px solid #F4A261;">
                        <p style="color: #4B2F1F; font-size: 16px; margin: 0 0 10px 0; font-weight: 500;">
                            Solde initial saisi:
                            <strong id="start-expected-amount" style="color: #F4A261; font-size: 18px;">0 BIF</strong>
                        </p>
                        <small style="display: block; color: #4B2F1F;">Comptez le montant total des espèces et confirmez qu'il correspond au solde initial.</small>
                    </div>
                    <div style="margin-bottom: 25px;">
                        <label for="start-cash-count" style="color: #4B2F1F; font-size: 16px; font-weight: 600; display: block; margin-bottom: 10px;">Montant Total Compté (BIF)</label>
                        <input
                            type="number"
                            id="start-cash-count"
                            min="0"
                            step="1"
                            inputmode="numeric"
                            style="width: 100%; padding: 14px; border: 2px solid #F4F0E4; border-radius: 8px; font-size: 18px; font-weight: 600; text-align: center; transition: border-color 0.3s, box-shadow 0.3s;"
                            placeholder="0"
                            onfocus="this.style.borderColor='#F4A261'; this.style.boxShadow='0 0 0 3px rgba(244, 162, 97, 0.1)';"
                            onblur="this.style.borderColor='#F4F0E4'; this.style.boxShadow='none';"
                        >
                    </div>
                    <div style="background: linear-gradient(135deg, #4B2F1F 0%, #F4A261 100%); padding: 20px; border-radius: 12px; margin-bottom: 20px; color: #F4F0E4;">
                        <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                            <span style="font-size: 16px; font-weight: 600;">Résumé</span>
                            <div style="width: 30px; height: 2px; background-color: #F4F0E4; border-radius: 1px;"></div>
                        </div>
                        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px;">
                            <div>
                                <div style="font-size: 13px; opacity: 0.9; margin-bottom: 4px;">Total Compté</div>
                                <div style="font-size: 18px; font-weight: 700;" id="start-counted-total">0 BIF</div>
                            </div>
                            <div>
                                <div style="font-size: 13px; opacity: 0.9; margin-bottom: 4px;">Différence</div>
                                <div style="font-size: 18px; font-weight: 700;" id="start-difference-total">0 BIF</div>
                            </div>
                        </div>
                    </div>
                    <div id="start-cash-status" style="font-size: 14px; margin-bottom: 20px; padding: 12px; border-radius: 8px; text-align: center; font-weight: 500; background-color: transparent;"></div>
                    <div style="display: flex; gap: 15px; justify-content: flex-end; padding-top: 10px; border-top: 1px solid #F4F0E4;">
                        <button type="button" id="cancel-start-cash" style="padding: 12px 24px; background-color: #F4F0E4; color: #4B2F1F; border: 2px solid #4B2F1F; border-radius: 8px; font-size: 14px; font-weight: 600; cursor: pointer; transition: all 0.3s;">Annuler</button>
                        <button type="button" id="confirm-start-cash" style="padding: 12px 24px; background: linear-gradient(135deg, #4B2F1F 0%, #F4A261 100%); color: #F4F0E4; border: none; border-radius: 8px; font-size: 14px; font-weight: 600; cursor: pointer; transition: all 0.3s; box-shadow: 0 4px 12px rgba(75, 47, 31, 0.3);" disabled>Confirmer & Démarrer Session</button>
                    </div>
                    <input type="hidden" id="start_cash_total" name="start_cash_total" value="">
                </div>
            </div>
        <?php endif; ?>
        <h3 style="color: #4B2F1F; margin-top: 20px; margin-bottom: 10px;">Sessions Récentes</h3>
        <table style="width: 100%; border-collapse: collapse; background-color: #FFFFFF; border-radius: 8px; overflow: hidden; margin-bottom: 20px;">
            <thead>
                <tr style="background-color: #4B2F1F; color: #F4F0E4;">
                    <th style="padding: 12px; text-align: left;">Début Session</th>
                    <th style="padding: 12px; text-align: left;">Fin Session</th>
                    <th style="padding: 12px; text-align: left;">Solde Initial</th>
                    <th style="padding: 12px; text-align: left;">Dépôts</th>
                    <th style="padding: 12px; text-align: left;">Retraits</th>
                    <th style="padding: 12px; text-align: left;">Solde Final</th>
                    <th style="padding: 12px; text-align: left;">Montant</th>
                    <th style="padding: 12px; text-align: left;">Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php if (!empty($recent_shifts)): ?>
                    <?php foreach ($recent_shifts as $shift): ?>
                        <?php
                            $counted_amount = isset($shift['counted_amount']) ? (int)$shift['counted_amount'] : null;
                        ?>
                        <tr style="border-bottom: 1px solid #F4F0E4;">
                            <td style="padding: 12px;"><?php echo date('Y-m-d H:i', strtotime($shift['shift_start'])); ?></td>
                            <td style="padding: 12px;"><?php echo $shift['shift_end'] ? date('Y-m-d H:i', strtotime($shift['shift_end'])) : 'N/A'; ?></td>
                            <td style="padding: 12px;"><?php echo number_format((int)$shift['beginning_balance'], 0, ',', '.') . ' BIF'; ?></td>
                            <td style="padding: 12px;"><?php echo number_format((int)$shift['deposits'], 0, ',', '.') . ' BIF'; ?></td>
                            <td style="padding: 12px;"><?php echo number_format((int)$shift['withdrawals'], 0, ',', '.') . ' BIF'; ?></td>
                            <td style="padding: 12px;"><?php echo $shift['end_balance'] ? number_format((int)$shift['end_balance'], 0, ',', '.') . ' BIF' : 'N/A'; ?></td>
                            <td style="padding: 12px;"><?php echo $counted_amount !== null ? number_format($counted_amount, 0, ',', '.') . ' BIF' : 'N/A'; ?></td>
                            <td style="padding: 12px;">
                                <button type="button" class="view-shift-btn" data-shift-modal="shift-details-<?php echo (int)$shift['id']; ?>" style="padding: 8px 14px; background-color: #4B2F1F; color: #F4F0E4; border: none; border-radius: 6px; cursor: pointer;">Voir</button>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                <?php else: ?>
                    <tr>
                        <td colspan="8" style="padding: 12px; text-align: center; color: #4B2F1F;">Aucune session enregistrée pour le moment.</td>
                    </tr>
                <?php endif; ?>
            </tbody>
        </table>
        <?php if (!empty($recent_shifts)): ?>
            <?php foreach ($recent_shifts as $shift): 
                $modal_id = 'shift-details-' . (int)$shift['id'];
                $shift_id = (int)$shift['id'];
                $counted = isset($shift['counted_amount']) ? (int)$shift['counted_amount'] : null;
                $cashier_display_name = $shift['cashier_name'] ?? ($_SESSION['full_name'] ?? 'Unknown');
                $modal_province_name = $shift['province_name'] ?? $province_name ?? 'Unknown';
                $closure_recorded_raw = $shift['closure_recorded_at'] ?? null;
                $closure_recorded_display = 'N/A';
                if ($closure_recorded_raw) {
                    $closure_timestamp = strtotime($closure_recorded_raw);
                    if ($closure_timestamp !== false) {
                        $closure_recorded_display = date('Y-m-d H:i:s', $closure_timestamp);
                    }
                }
                
                // Get pre-calculated invoice payments and loan repayments for this shift
                $invoice_payments = $shift_invoice_payments[$shift_id] ?? 0;
                $loan_repayments = $recent_shift_loan_repayments[$shift_id] ?? 0;
            ?>
                <div id="<?php echo htmlspecialchars($modal_id, ENT_QUOTES, 'UTF-8'); ?>" class="shift-modal" style="display: none;">
                    <div class="shift-modal-content" style="background-color: #FFFFFF; padding: 24px; border-radius: 12px; max-width: 600px; width: 100%; box-shadow: 0 10px 30px rgba(0,0,0,0.35); position: relative;">
                        <button type="button" class="shift-modal-close" style="position: absolute; top: 12px; right: 12px; background: none; border: none; font-size: 20px; cursor: pointer;">&times;</button>
                        <h3 style="color: #4B2F1F; margin-top: 0;">Détails Session</h3>
                        <p style="margin: 6px 0;"><strong>Caissier:</strong> <?php echo htmlspecialchars($cashier_display_name, ENT_QUOTES, 'UTF-8'); ?></p>
                        <p style="margin: 6px 0;"><strong>Dépôt:</strong> <?php echo htmlspecialchars($modal_province_name, ENT_QUOTES, 'UTF-8'); ?></p>
                        <p style="margin: 6px 0;"><strong>Début Session:</strong> <?php echo date('Y-m-d H:i', strtotime($shift['shift_start'])); ?></p>
                        <p style="margin: 6px 0;"><strong>Fin Session:</strong> <?php echo $shift['shift_end'] ? date('Y-m-d H:i', strtotime($shift['shift_end'])) : 'Active'; ?></p>
                        <p style="margin: 6px 0;"><strong>Solde Initial:</strong> <?php echo number_format((int)$shift['beginning_balance'], 0, ',', '.'); ?> BIF</p>
                        <p style="margin: 6px 0;"><strong>Ventes:</strong> <?php echo number_format($invoice_payments, 0, ',', '.'); ?> BIF</p>
                        <p style="margin: 6px 0;"><strong>Remboursements Crédits:</strong> <?php echo number_format($loan_repayments, 0, ',', '.'); ?> BIF</p>
                        <p style="margin: 6px 0;"><strong>Retraits:</strong> <?php echo number_format((int)$shift['withdrawals'], 0, ',', '.'); ?> BIF</p>
                        <p style="margin: 6px 0;"><strong>Montant:</strong> <?php echo $counted !== null ? number_format($counted, 0, ',', '.') . ' BIF' : 'N/A'; ?></p>
                        <p style="margin: 12px 0 0; font-size: 13px; color: #4B2F1F;"><strong>PDF Généré:</strong> <?php echo htmlspecialchars($closure_recorded_display, ENT_QUOTES, 'UTF-8'); ?></p>
                        <div style="margin-top: 20px; display: flex; justify-content: flex-end; gap: 12px;">
                            <button type="button" class="shift-modal-print" data-print-target="<?php echo htmlspecialchars($modal_id, ENT_QUOTES, 'UTF-8'); ?>" style="padding: 10px 16px; background-color: #4B2F1F; color: #F4F0E4; border: none; border-radius: 8px; cursor: pointer;">Imprimer / Sauvegarder PDF</button>
                            <button type="button" class="shift-modal-close" style="padding: 10px 16px; background-color: #F4A261; color: #4B2F1F; border: none; border-radius: 8px; cursor: pointer;">Fermer</button>
                        </div>
                        <p style="margin-top: 15px; font-size: 12px; color: #4B2F1F; text-align: right;">Imprimé à: <span class="print-timestamp" data-default="<?php echo htmlspecialchars($closure_recorded_display, ENT_QUOTES, 'UTF-8'); ?>"><?php echo htmlspecialchars($closure_recorded_display, ENT_QUOTES, 'UTF-8'); ?></span></p>
                    </div>
                </div>
            <?php endforeach; ?>
        <?php endif; ?>
        <div style="margin-top: 20px; text-align: center;">
            <a href="/masunzu_bar_hotel/dashboards/cashier_dashboard.php" style="background-color: #F4A261; color: #000000; padding: 10px 20px; border-radius: 8px; text-decoration: none; font-weight: bold;">Retour au Tableau de Bord</a>
        </div>
    </div>
</div>
<?php if ($active_shift): ?>
<div id="cash-count-modal" class="cash-modal" style="display: none; align-items: flex-start; overflow-y: auto;">
    <div class="cash-modal-content" style="background: linear-gradient(135deg, #FFFFFF 0%, #F4F0E4 100%); padding: 30px; border-radius: 16px; width: 100%; max-width: 580px; box-shadow: 0 12px 40px rgba(75, 47, 31, 0.3); border: 2px solid #F4A261; margin-top: 20px; margin-bottom: 20px;">
        <div style="text-align: center; margin-bottom: 25px;">
            <h3 style="color: #4B2F1F; margin: 0 0 8px 0; font-size: 24px; font-weight: 600;">Confirmer Espèces de Clôture</h3>
            <div style="width: 60px; height: 3px; background: linear-gradient(90deg, #4B2F1F 0%, #F4A261 100%); margin: 0 auto; border-radius: 2px;"></div>
        </div>
        <div style="background-color: rgba(75, 47, 31, 0.05); padding: 20px; border-radius: 12px; margin-bottom: 25px; border-left: 4px solid #F4A261;">
            <p style="color: #4B2F1F; font-size: 16px; margin: 0; font-weight: 500;">Espèces attendues à compter: <strong style="color: #F4A261; font-size: 18px;"><?php echo number_format((int)$current_balance, 0, ',', '.'); ?> BIF</strong></p>
        </div>
        <div style="margin-bottom: 25px;">
            <label style="color: #4B2F1F; font-size: 16px; font-weight: 600; display: block; margin-bottom: 10px;">Total Espèces Comptées (BIF)</label>
            <input 
                type="number" 
                min="0" 
                step="1"
                id="cash-count-input" 
                style="width: 100%; padding: 15px; border: 2px solid #F4F0E4; border-radius: 8px; font-size: 18px; font-weight: 600; text-align: center; transition: border-color 0.3s, box-shadow 0.3s;" 
                placeholder="0" 
                onfocus="this.style.borderColor='#F4A261'; this.style.boxShadow='0 0 0 3px rgba(244, 162, 97, 0.1)';" 
                onblur="this.style.borderColor='#F4F0E4'; this.style.boxShadow='none';"
            >
        </div>
        <div style="background: linear-gradient(135deg, #4B2F1F 0%, #F4A261 100%); padding: 20px; border-radius: 12px; margin-bottom: 20px; color: #F4F0E4;">
            <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                <span style="font-size: 16px; font-weight: 600;">Résumé</span>
                <div style="width: 30px; height: 2px; background-color: #F4F0E4; border-radius: 1px;"></div>
            </div>
            <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px;">
                <div>
                    <div style="font-size: 13px; opacity: 0.9; margin-bottom: 4px;">Total Compté</div>
                    <div style="font-size: 18px; font-weight: 700;" id="counted-total">0 BIF</div>
                </div>
                <div>
                    <div style="font-size: 13px; opacity: 0.9; margin-bottom: 4px;">Différence</div>
                    <div style="font-size: 18px; font-weight: 700;" id="difference-total">0 BIF</div>
                </div>
            </div>
        </div>

        <div style="margin-bottom: 20px;">
            <label style="color: #4B2F1F; font-size: 14px; font-weight: 600; display: block; margin-bottom: 8px;">Notes Session (Optionnel)</label>
            <textarea id="shift-notes" placeholder="Ajouter des notes sur cette session..." style="width: 100%; padding: 12px; border: 2px solid #F4F0E4; border-radius: 8px; font-size: 14px; min-height: 70px; resize: vertical; background-color: #FFFFFF; transition: border-color 0.3s, box-shadow 0.3s;" onfocus="this.style.borderColor='#F4A261'; this.style.boxShadow='0 0 0 3px rgba(244, 162, 97, 0.1)';" onblur="this.style.borderColor='#F4F0E4'; this.style.boxShadow='none';"></textarea>
        </div>

        <div id="cash-status-message" style="font-size: 14px; margin-bottom: 20px; padding: 12px; border-radius: 8px; text-align: center; font-weight: 500;"></div>

        <div style="display: flex; gap: 15px; justify-content: flex-end; padding-top: 10px; border-top: 1px solid #F4F0E4;">
            <button type="button" id="cancel-cash-count" style="padding: 12px 24px; background-color: #F4F0E4; color: #4B2F1F; border: 2px solid #4B2F1F; border-radius: 8px; font-size: 14px; font-weight: 600; cursor: pointer; transition: all 0.3s;">Annuler</button>
            <button type="button" id="confirm-cash-count" style="padding: 12px 24px; background: linear-gradient(135deg, #4B2F1F 0%, #F4A261 100%); color: #F4F0E4; border: none; border-radius: 8px; font-size: 14px; font-weight: 600; cursor: pointer; transition: all 0.3s; box-shadow: 0 4px 12px rgba(75, 47, 31, 0.3);" disabled>Confirmer & Fermer Session</button>
        </div>
    </div>
</div>
<?php endif; ?>
<style>
    .cash-modal {
        position: fixed;
        inset: 0;
        background-color: rgba(0,0,0,0.6);
        display: flex;
        align-items: center;
        justify-content: center;
        z-index: 2000;
        padding: 20px;
    }
    @keyframes blink {
        0% { opacity: 1; }
        50% { opacity: 0.5; }
        100% { opacity: 1; }
    }
    .blink {
        animation: blink 1s infinite;
    }
    .shift-modal {
        position: fixed;
        inset: 0;
        background-color: rgba(0,0,0,0.55);
        display: none;
        align-items: center;
        justify-content: center;
        z-index: 2100;
        padding: 20px;
    }
</style>
<script>
document.addEventListener('DOMContentLoaded', function () {
    const openModalBtn = document.getElementById('open-cash-count-modal');
    const modal = document.getElementById('cash-count-modal');
    document.querySelectorAll('.print-timestamp').forEach(function (span) {
        if (span.dataset.default) {
            span.textContent = span.dataset.default;
        }
    });
    if (openModalBtn && modal) {
        const form = document.getElementById('end-shift-form');
        const countedTotalEl = document.getElementById('counted-total');
        const differenceEl = document.getElementById('difference-total');
        const statusEl = document.getElementById('cash-status-message');
        const confirmBtn = document.getElementById('confirm-cash-count');
        const cancelBtn = document.getElementById('cancel-cash-count');
        const cashCountInput = document.getElementById('cash-count-input');
        const expectedCash = <?php echo (int)$current_balance; ?>;
        const hiddenTotal = document.getElementById('cash_count_total');
        const hiddenDifference = document.getElementById('cash_difference');
        const hiddenNotes = document.getElementById('notes_hidden');
        const notesTextarea = document.getElementById('shift-notes');

        function formatAmount(value) {
            return (value || 0).toLocaleString('en-US') + ' BIF';
        }

        function computeTotals() {
            const total = parseInt(cashCountInput.value, 10) || 0;
            const difference = total - expectedCash;
            countedTotalEl.textContent = formatAmount(total);
            differenceEl.textContent = formatAmount(difference);

            let differenceColor = '#000000';
            if (difference < 0) {
                differenceColor = '#B00020';
            } else if (difference === 0) {
                differenceColor = '#2E7D32';
            } else {
                differenceColor = '#B00020';
            }
            differenceEl.style.color = differenceColor;

            let message = '';
            let messageColor = '#4B2F1F';
            let canConfirm = false;

            // Special case: If expectedCash is 0, allow closing with total of 0
            if (expectedCash === 0 && total === 0) {
                message = 'Aucun paiement n\'a été effectué pendant cette session. Vous pouvez fermer la session avec zéro espèces.';
                messageColor = '#2E7D32';
                canConfirm = true;
            } else if (expectedCash === 0 && total > 0) {
                message = 'Aucun paiement n\'a été effectué pendant cette session (attendu: 0 BIF). Le comptage d\'espèces dépasse le montant attendu. Aucun excédent n\'est autorisé.';
                messageColor = '#B00020';
            } else if (total === 0 && expectedCash > 0) {
                message = 'Entrez le total des espèces comptées.';
                messageColor = '#B00020';
            } else if (total < expectedCash) {
                message = 'Le comptage d\'espèces est inférieur au montant attendu. Veuillez recomptez.';
                messageColor = '#B00020';
            } else if (difference > 0) {
                message = 'Le comptage d\'espèces dépasse le montant attendu. Aucun excédent n\'est autorisé. Veuillez vérifier et ajuster.';
                messageColor = '#B00020';
            } else if (difference === 0) {
                message = 'Le comptage correspond exactement au montant attendu. Vous pouvez fermer la session.';
                messageColor = '#2E7D32';
                canConfirm = true;
            } else {
                message = 'Le comptage d\'espèces est inférieur au montant attendu. Veuillez recomptez.';
                messageColor = '#B00020';
            }

            statusEl.textContent = message;
            statusEl.style.color = messageColor;

            if (message.includes('Entrez le total')) {
                statusEl.style.backgroundColor = '#FFF3CD';
                statusEl.style.border = '1px solid #FFEAA7';
            } else if (message.includes('inférieur') || message.includes('dépasse') || message.includes('recomptez')) {
                statusEl.style.backgroundColor = '#F8D7DA';
                statusEl.style.border = '1px solid #F5C6CB';
            } else if (canConfirm) {
                statusEl.style.backgroundColor = '#D4EDDA';
                statusEl.style.border = '1px solid #C3E6CB';
            } else {
                statusEl.style.backgroundColor = 'transparent';
                statusEl.style.border = 'none';
            }

            confirmBtn.disabled = !canConfirm;

            return { total, difference };
        }

        if (cashCountInput) {
            cashCountInput.addEventListener('input', computeTotals);
        }

                        openModalBtn.addEventListener('click', function (event) {
                            event.preventDefault();
                            event.stopPropagation();
                            modal.style.display = 'flex';
                            computeTotals();
                        });

                        cancelBtn.addEventListener('click', function () {
                            modal.style.display = 'none';
                        });

                        confirmBtn.addEventListener('click', function () {
                            const { total, difference } = computeTotals();
                            if (confirmBtn.disabled) {
                                return;
                            }
                            hiddenTotal.value = total;
                            hiddenDifference.value = difference;
                            if (hiddenNotes && notesTextarea) {
                                hiddenNotes.value = notesTextarea.value.trim();
                            }

                            let endShiftInput = form.querySelector('input[name="end_shift"]');
                            if (!endShiftInput) {
                                endShiftInput = document.createElement('input');
                                endShiftInput.type = 'hidden';
                                endShiftInput.name = 'end_shift';
                                endShiftInput.value = '1';
                                form.appendChild(endShiftInput);
                            }

                            modal.style.display = 'none';
                            form.submit();
                        });

                        modal.addEventListener('click', function (event) {
                            if (event.target === modal) {
                                modal.style.display = 'none';
                            }
                        });
                    }

    const startForm = document.getElementById('start-shift-form');
    const startModal = document.getElementById('start-cash-modal');
    if (startForm && startModal) {
        const beginningInput = document.getElementById('beginning_balance_input');
        const startOpenBtn = document.getElementById('open-start-cash-modal');
        const startCashCountInput = document.getElementById('start-cash-count');
        const startTotalEl = document.getElementById('start-counted-total');
        const startDifferenceEl = document.getElementById('start-difference-total');
        const startStatusEl = document.getElementById('start-cash-status');
        const startExpectedEl = document.getElementById('start-expected-amount');
        const startConfirmBtn = document.getElementById('confirm-start-cash');
        const startCancelBtn = document.getElementById('cancel-start-cash');
        const startHiddenTotal = document.getElementById('start_cash_total');

        if (!beginningInput || !startOpenBtn || !startCashCountInput || !startTotalEl || !startDifferenceEl || !startStatusEl || !startExpectedEl || !startConfirmBtn || !startCancelBtn || !startHiddenTotal) {
            console.error('Missing required elements for start shift form');
        } else {

        function ensureStartShiftFlag() {
            let startFlag = startForm.querySelector('input[name="start_shift"]');
            if (!startFlag) {
                startFlag = document.createElement('input');
                startFlag.type = 'hidden';
                startFlag.name = 'start_shift';
                startFlag.value = '1';
                startForm.appendChild(startFlag);
            }
        }

        function formatCurrency(amount) {
            return amount.toLocaleString('en-US') + ' BIF';
        }

        function computeStartTotals() {
            const total = parseInt(startCashCountInput.value, 10) || 0;
            const expected = parseInt(startModal.dataset.expected || '0', 10) || 0;
            const difference = total - expected;
            
            startTotalEl.textContent = formatCurrency(total);
            startDifferenceEl.textContent = formatCurrency(difference);

            let canConfirm = false;
            if (expected > 0) {
                if (total === expected) {
                    startStatusEl.textContent = 'Le montant compté correspond au solde initial.';
                    startStatusEl.style.color = '#155724';
                    startStatusEl.style.backgroundColor = '#D4EDDA';
                    startStatusEl.style.border = '1px solid #C3E6CB';
                    canConfirm = true;
                } else if (total < expected) {
                    startStatusEl.textContent = 'Le montant compté est inférieur au solde initial.';
                    startStatusEl.style.color = '#B00020';
                    startStatusEl.style.backgroundColor = '#F8D7DA';
                    startStatusEl.style.border = '1px solid #F5C6CB';
                } else {
                    startStatusEl.textContent = 'Le montant compté est supérieur au solde initial.';
                    startStatusEl.style.color = '#B00020';
                    startStatusEl.style.backgroundColor = '#F8D7DA';
                    startStatusEl.style.border = '1px solid #F5C6CB';
                }
            } else {
                startStatusEl.textContent = 'Le solde initial est zéro. Aucun comptage requis.';
                startStatusEl.style.color = '#4B2F1F';
                startStatusEl.style.backgroundColor = '#FFF3CD';
                startStatusEl.style.border = '1px solid #FFEAA7';
                canConfirm = true;
            }

            startConfirmBtn.disabled = !canConfirm;
            return { total, difference };
        }

        function openStartModal(beginningValue) {
            startModal.dataset.expected = beginningValue.toString();
            startExpectedEl.textContent = formatCurrency(beginningValue);
            
            if (beginningValue > 0) {
                startStatusEl.textContent = 'Comptez le montant total des espèces et entrez-le ci-dessous.';
                startStatusEl.style.color = '#4B2F1F';
                startStatusEl.style.backgroundColor = '#FFF3CD';
                startStatusEl.style.border = '1px solid #FFEAA7';
            } else {
                startStatusEl.textContent = 'Le solde initial est zéro. Aucun comptage requis.';
                startStatusEl.style.color = '#4B2F1F';
                startStatusEl.style.backgroundColor = '#FFF3CD';
                startStatusEl.style.border = '1px solid #FFEAA7';
            }
            
            startCashCountInput.value = '';
            startConfirmBtn.disabled = beginningValue > 0;
            computeStartTotals();
            startModal.style.display = 'flex';
            startCashCountInput.focus();
        }

        function handleStartAttempt() {
            const beginningValue = parseInt(beginningInput.value, 10) || 0;
            if (beginningValue <= 0) {
                startHiddenTotal.value = '';
                ensureStartShiftFlag();
                startForm.submit();
                return;
            }
            openStartModal(beginningValue);
        }

        startCashCountInput.addEventListener('input', computeStartTotals);

        startCancelBtn.addEventListener('click', function () {
            startModal.style.display = 'none';
            startConfirmBtn.disabled = true;
            startCashCountInput.value = '';
            startHiddenTotal.value = '';
        });

        startConfirmBtn.addEventListener('click', function () {
            const { total } = computeStartTotals();
            const expected = parseInt(startModal.dataset.expected || '0', 10) || 0;
            if (expected > 0 && total !== expected) {
                return;
            }
            startHiddenTotal.value = total;
            startModal.style.display = 'none';
            ensureStartShiftFlag();
            startForm.submit();
        });

        startForm.addEventListener('submit', function (event) {
            event.preventDefault();
            handleStartAttempt();
        });

        if (startOpenBtn) {
            startOpenBtn.addEventListener('click', function (event) {
                event.preventDefault();
                event.stopPropagation();
                handleStartAttempt();
            });
        } else {
            console.error('Start shift button not found');
        }

        startModal.addEventListener('click', function (event) {
            if (event.target === startModal) {
                startModal.style.display = 'none';
                startConfirmBtn.disabled = true;
            }
        });
        }
    } else {
        console.log('Start shift form or modal not found - may not be visible on this page');
    }

    const viewButtons = document.querySelectorAll('.view-shift-btn');
    viewButtons.forEach(function (btn) {
        btn.addEventListener('click', function () {
            const targetId = btn.getAttribute('data-shift-modal');
            const targetModal = document.getElementById(targetId);
            if (targetModal) {
                targetModal.style.display = 'flex';
            }
        });
    });

    document.querySelectorAll('.shift-modal-close').forEach(function (btn) {
        btn.addEventListener('click', function () {
            const modalEl = btn.closest('.shift-modal');
            if (modalEl) {
                modalEl.style.display = 'none';
            }
        });
    });

    document.querySelectorAll('.shift-modal').forEach(function (modalEl) {
        modalEl.addEventListener('click', function (event) {
            if (event.target === modalEl) {
                modalEl.style.display = 'none';
            }
        });
    });

    document.querySelectorAll('.shift-modal-print').forEach(function (btn) {
        btn.addEventListener('click', function () {
            const targetId = btn.getAttribute('data-print-target');
            const targetModal = document.getElementById(targetId);
            if (!targetModal) {
                return;
            }
            const content = targetModal.querySelector('.shift-modal-content').cloneNode(true);
            if (content) {
                const timestampEl = content.querySelector('.print-timestamp');
                const sourceTimestampEl = targetModal.querySelector('.print-timestamp');
                if (timestampEl) {
                    const now = new Date();
                    const timestampValue = now.toLocaleString();
                    timestampEl.textContent = timestampValue;
                    if (sourceTimestampEl) {
                        sourceTimestampEl.textContent = timestampValue;
                        sourceTimestampEl.dataset.default = timestampValue;
                    }
                }
                content.querySelectorAll('button').forEach(function (button) {
                    button.remove();
                });
                const printWindow = window.open('', '_blank', 'width=800,height=600');
                if (printWindow) {
                    printWindow.document.write('<html><head><title>Détails Session</title>');
                    printWindow.document.write('<style>body{font-family:Arial,sans-serif;color:#4B2F1F;padding:20px;} table{width:100%;border-collapse:collapse;} th,td{padding:8px;border:1px solid #ddd;} h3{margin-top:0;}</style>');
                    printWindow.document.write('</head><body>');
                    printWindow.document.body.appendChild(content);
                    printWindow.document.write('</body></html>');
                    printWindow.document.close();
                    printWindow.focus();
                    printWindow.print();
                }
            }
        });
    });
});
</script>
<?php require_once __DIR__ . '/../../includes/footer.php'; ?>
