<?php
session_start();
require_once __DIR__ . '/../../includes/auth.php';
require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../includes/functions.php';

require_login();

header('Content-Type: application/json');

$conn = get_db_connection();
if (!$conn) {
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => 'Database connection failed']);
    exit;
}

try {
    $payment_id = intval($_GET['payment_id'] ?? 0);
    if ($payment_id <= 0) {
        http_response_code(400);
        echo json_encode(['success' => false, 'error' => 'Invalid payment ID']);
        exit;
    }

    // Fetch payment details from paid_invoices
    $stmt = $conn->prepare("
        SELECT 
            pi.id,
            pi.invoice_number,
            pi.customer_name,
            pi.phone_number,
            pi.paid_amount,
            pi.subtotal_amount,
            pi.tax_amount,
            pi.tax_rate,
            pi.tax_mode,
            pi.payment_method,
            pi.payment_details,
            pi.paid_at,
            pi.nif,
            pi.rc,
            pi.is_loan_sale,
            r.receipt_id
        FROM paid_invoices pi
        LEFT JOIN receipts r ON r.transaction_type = 'invoice' AND r.transaction_id = pi.id
        WHERE pi.id = ? AND r.created_by = ?
        LIMIT 1
    ");
    if (!$stmt) {
        throw new Exception("Prepare failed: " . $conn->error);
    }
    $stmt->bind_param("ii", $payment_id, $_SESSION['user_id']);
    $stmt->execute();
    $payment = $stmt->get_result()->fetch_assoc();
    $stmt->close();

    if (!$payment) {
        http_response_code(404);
        echo json_encode(['success' => false, 'error' => 'Payment not found or unauthorized']);
        exit;
    }

    // Fetch payment items from paid_invoice_items
    $items = [];
    $stmt = $conn->prepare("
        SELECT 
            p.name AS product_name,
            pii.quantity,
            pii.unit_price AS price_per_unit
        FROM paid_invoice_items pii
        JOIN product p ON pii.product_id = p.id
        WHERE pii.invoice_id = ?
        ORDER BY pii.id
    ");
    if (!$stmt) {
        throw new Exception("Prepare failed for items: " . $conn->error);
    }
    $stmt->bind_param("i", $payment_id);
    $stmt->execute();
    $items_result = $stmt->get_result();
    while ($row = $items_result->fetch_assoc()) {
        $items[] = $row;
    }
    $stmt->close();

    $payment['items'] = $items;

    echo json_encode([
        'success' => true,
        'payment' => $payment
    ]);

    $conn->close();
} catch (Exception $e) {
    error_log("Error in get_payment_details.php: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => 'Server error: ' . $e->getMessage()]);
    if ($conn) $conn->close();
}
?>
