<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

$base_dir = '/opt/lampp/htdocs/masunzu_bar_hotel';
$auth_path = $base_dir . '/includes/auth.php';
$database_path = $base_dir . '/config/database.php';
$functions_path = $base_dir . '/includes/functions.php';
$footer_path = $base_dir . '/includes/footer.php';
$header_path = $base_dir . '/includes/header.php';

if (!file_exists($auth_path) || !file_exists($database_path) || !file_exists($functions_path)) {
    error_log("ERROR: Missing critical file - auth: " . file_exists($auth_path) . ", database: " . file_exists($database_path) . ", functions: " . file_exists($functions_path));
    die("Erreur: Fichier requis manquant. Contactez l'administrateur.");
}

require_once $auth_path;
require_once $database_path;
require_once $functions_path;

/**
 * Return the name of a province for a given ID.
 */
function getProvinceNameById(mysqli $conn, ?int $province_id): string {
    if (empty($province_id) || $province_id <= 0) {
        return 'Unknown';
    }
    $stmt = $conn->prepare("SELECT name FROM province WHERE id = ?");
    if (!$stmt) {
        return 'Unknown';
    }
    $stmt->bind_param("i", $province_id);
    if (!$stmt->execute()) {
        $stmt->close();
        return 'Unknown';
    }
    $result = $stmt->get_result();
    $row = $result ? $result->fetch_assoc() : null;
    $stmt->close();
    $label = $row['name'] ?? '';
    return $label !== '' ? $label : 'Unknown';
}

try {
    error_log("DEBUG: expenses/create.php loaded at " . date('Y-m-d H:i:s') . ", user_id=" . ($_SESSION['user_id'] ?? 'unset') . ", method=" . $_SERVER['REQUEST_METHOD'] . ", raw_post=" . json_encode($_POST));

    require_login();

    $conn = get_db_connection();
    if (!$conn) {
        throw new Exception("Échec de la connexion à la base de données: " . mysqli_connect_error());
    }

    $user_id = isset($_SESSION['user_id']) ? (int)$_SESSION['user_id'] : null;
    $role_id = isset($_SESSION['role_id']) ? (int)$_SESSION['role_id'] : null;
    $role_name = $_SESSION['role_name'] ?? '';
    $full_name = $_SESSION['full_name'] ?? '';

    // Fetch province_id from session or database
    $province_id = null;
    if (isset($_SESSION['province_id']) && $_SESSION['province_id'] > 0) {
        $province_id = (int)$_SESSION['province_id'];
    } else {
        // Fetch province_id from database if not in session
        $stmt = $conn->prepare("SELECT province_id FROM user WHERE id = ?");
        if ($stmt) {
            $stmt->bind_param("i", $user_id);
            $stmt->execute();
            $result = $stmt->get_result();
            $user_row = $result->fetch_assoc();
            if ($user_row && isset($user_row['province_id']) && $user_row['province_id'] > 0) {
                $province_id = (int)$user_row['province_id'];
                $_SESSION['province_id'] = $province_id; // Update session for future use
            }
            $stmt->close();
        }
    }

    // All authenticated users can initiate a requisition; approvals remain DG/DGA.

    $province = getProvinceNameById($conn, $province_id);

    $provinceOptions = [];
    $provinceStmt = $conn->prepare("SELECT id, name FROM province ORDER BY name ASC");
    if ($provinceStmt) {
        if ($provinceStmt->execute()) {
            $provinceResult = $provinceStmt->get_result();
            if ($provinceResult) {
                while ($row = $provinceResult->fetch_assoc()) {
                    $provinceOptions[] = [
                        'id' => (int)($row['id'] ?? 0),
                        'name' => $row['name'] ?? '',
                    ];
                }
                $provinceResult->free();
            }
        }
        $provinceStmt->close();
    }

    $cashiers = [];
    $cashierStmt = $conn->prepare("SELECT id, full_name FROM user WHERE role_id = (SELECT id FROM role WHERE name = 'Caissier_Comptable') ORDER BY full_name ASC");
    if ($cashierStmt) {
        $cashierStmt->execute();
        $cashierResult = $cashierStmt->get_result();
        while ($cashierRow = $cashierResult->fetch_assoc()) {
            $cashiers[] = $cashierRow;
        }
        $cashierStmt->close();
    }
    $cashier_ids = array_map('intval', array_column($cashiers, 'id'));
    $requires_requisition_cashier = in_array($role_name, ['DG', 'DGA', 'Admin', 'Marketing Agent', 'Agent Marketing', 'Chef Marketing', 'Operation Manager'], true) || empty($province_id);
    $autoApproveRequisition = in_array($role_name, ['DG', 'DGA', 'Admin'], true);

    // Fetch all users for recipient selection (excluding the logged-in user)
    $users = [];
    try {
        $stmt = $conn->prepare("SELECT id, full_name FROM user WHERE username <> 'admin' ORDER BY full_name ASC");
        if (!$stmt) {
            error_log("ERROR: Query preparation failed for users: " . $conn->error);
            throw new Exception("Query preparation failed for users: " . $conn->error);
        }
        if (!$stmt->execute()) {
            error_log("ERROR: Query execution failed for users: " . $stmt->error);
            throw new Exception("Query execution failed for users: " . $stmt->error);
        }
        $result = $stmt->get_result();
        if ($result) {
            while ($row = $result->fetch_assoc()) {
                if (isset($row['id']) && isset($row['full_name'])) {
                    $users[] = $row;
                }
            }
            $result->free();
        }
        $stmt->close();
        
        // Debug: Log the number of users found (admin hidden only)
        error_log("DEBUG: Found " . count($users) . " users for recipient dropdown (excluding super admin)");
    } catch (Exception $e) {
        error_log("ERROR: Failed to fetch users: " . $e->getMessage());
        $users = []; // Ensure $users is set even on error
    }

    $categories = ['communication', 'transport', 'frais_de_mission', 'office_utilities', 'maintenance', 'others'];
    $selectedRecipientId = (int)$user_id;

    $success = isset($_GET['success']) ? htmlspecialchars($_GET['success'], ENT_QUOTES, 'UTF-8') : '';
    $errors = [];
    if ($requires_requisition_cashier && empty($cashiers)) {
        $errors[] = 'Aucun caissier disponible pour assigner cette réquisition. Contactez l\'administrateur.';
    }
    $assigned_cashier_to_save = $requires_requisition_cashier ? null : $user_id;
    $selected_assigned_cashier_id = $assigned_cashier_to_save;

    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['submit_requisition'])) {
        error_log("DEBUG: Processing POST at " . date('Y-m-d H:i:s') . ", raw_post=" . json_encode($_POST));
        $items = isset($_POST['items']) && is_array($_POST['items']) ? $_POST['items'] : [];
        $recipient_type = strtolower(trim($_POST['recipient_type'] ?? 'internal')) === 'external' ? 'external' : 'internal';
        $recipient_id = $recipient_type === 'internal' ? filter_input(INPUT_POST, 'recipient_id', FILTER_VALIDATE_INT) : null;
        if ($recipient_id) {
            $selectedRecipientId = (int)$recipient_id;
        }
        $external_recipient_name = trim($_POST['external_recipient_name'] ?? '');
        $external_recipient_phone = trim($_POST['external_recipient_phone'] ?? '');
        $assigned_cashier_input = filter_input(INPUT_POST, 'assigned_cashier_id', FILTER_VALIDATE_INT);
        if ($requires_requisition_cashier) {
            if (!$assigned_cashier_input || !in_array($assigned_cashier_input, $cashier_ids, true)) {
                $errors[] = "Veuillez sélectionner un caissier valide pour cette réquisition.";
            } else {
                $assigned_cashier_to_save = $assigned_cashier_input;
                $selected_assigned_cashier_id = $assigned_cashier_input;
            }
        } else {
            $assigned_cashier_to_save = $user_id;
            $selected_assigned_cashier_id = $user_id;
        }
        $postedProvinceId = filter_input(INPUT_POST, 'province_id', FILTER_VALIDATE_INT);
        $validProvinceIds = array_column($provinceOptions, 'id');
        if ($postedProvinceId && in_array($postedProvinceId, $validProvinceIds, true)) {
            $province_id = $postedProvinceId;
            $_SESSION['province_id'] = $province_id;
            $province = getProvinceNameById($conn, $province_id);
        }

        // Validate province_id - allow POST value if user doesn't have one set
        if (empty($province_id) || !is_numeric($province_id) || $province_id <= 0) {
            // If user doesn't have province_id, they must select one from the form
            if ($postedProvinceId && in_array($postedProvinceId, $validProvinceIds, true)) {
                $province_id = $postedProvinceId;
                $_SESSION['province_id'] = $province_id;
                $province = getProvinceNameById($conn, $province_id);
            } else {
                $errors[] = "Erreur: Veuillez sélectionner une province dans le formulaire.";
                error_log("ERROR: Invalid or missing province_id=" . ($province_id ?? 'NULL') . ", postedProvinceId=" . ($postedProvinceId ?? 'NULL'));
            }
        }

        // Validate recipient_id
    if ($recipient_type === 'internal') {
        if (!$recipient_id || !array_filter($users, fn($user) => $user['id'] === $recipient_id)) {
            $errors[] = "Veuillez sélectionner un destinataire valide de votre province.";
            error_log("ERROR: Invalid recipient_id=$recipient_id");
        }
        $external_recipient_name = null;
        $external_recipient_phone = null;
    } else {
        // For external recipients, ensure recipient_id is null and clear any submitted value
        $recipient_id = null;
        $external_recipient_name = mb_substr($external_recipient_name, 0, 150);
        $external_recipient_phone = preg_replace('/\D+/', '', $external_recipient_phone);
        if ($external_recipient_name === '') {
            $errors[] = "Entrez le nom complet du destinataire externe.";
        }
        if ($external_recipient_phone === '' || !preg_match('/^[0-9]{8,15}$/', $external_recipient_phone)) {
            $errors[] = "Entrez un numéro de téléphone valide (8-15 chiffres) pour le destinataire externe.";
        }
    }

        if (empty($items)) {
            $errors[] = "Aucun article de dépense soumis. Veuillez ajouter au moins un article.";
            error_log("ERROR: Empty items array");
        } else {
            $total_amount = 0;
            $valid_items = [];
            foreach ($items as $index => $item) {
                $category = isset($item['category']) ? trim($item['category']) : '';
                $description = isset($item['description']) ? trim($item['description']) : '';
                $quantity = isset($item['quantity']) ? trim($item['quantity']) : '';
                $unit_price = isset($item['unit_price']) ? trim($item['unit_price']) : '';

                $quantity = is_numeric($quantity) ? (int)$quantity : false;
                $unit_price = is_numeric($unit_price) ? (float)$unit_price : false;

                error_log("DEBUG: Item $index: category=$category, description=$description, quantity=" . ($quantity === false ? 'false' : $quantity) . ", unit_price=" . ($unit_price === false ? 'false' : $unit_price));

                if (!in_array($category, $categories)) {
                    $errors[] = "Veuillez sélectionner une catégorie valide pour l'article " . ($index + 1) . ".";
                }
                if (empty($description) || ctype_space($description)) {
                    $errors[] = "Veuillez entrer une description valide pour l'article " . ($index + 1) . ".";
                }
                if ($quantity === false || $quantity <= 0) {
                    $errors[] = "Veuillez entrer une quantité positive pour l'article " . ($index + 1) . ".";
                }
                if ($unit_price === false || $unit_price <= 0) {
                    $errors[] = "Veuillez entrer un prix unitaire positif pour l'article " . ($index + 1) . ".";
                } else {
                    $item_total = $quantity * $unit_price;
                    $total_amount += $item_total;
                    $valid_items[] = [
                        'category' => $category,
                        'description' => htmlspecialchars($description, ENT_QUOTES, 'UTF-8'),
                        'quantity' => $quantity,
                        'unit_price' => $unit_price
                    ];
                }
            }

            if (empty($valid_items)) {
                $errors[] = "Aucun article de dépense valide fourni. Veuillez vérifier vos entrées.";
                error_log("ERROR: No valid items, errors=" . json_encode($errors));
            }
        }

        if (empty($errors)) {
            // Ensure AUTO_INCREMENT is set on requisition.id and requisition_item.id
            // This is a one-time migration that will modify the schema if needed
            // NOTE: This must be done BEFORE beginning the transaction, as ALTER TABLE causes implicit commits
            $check_auto_inc_sql = "SELECT COLUMN_NAME, EXTRA FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'requisition' AND COLUMN_NAME = 'id'";
            $auto_inc_check = $conn->query($check_auto_inc_sql);
            if ($auto_inc_check) {
                $auto_inc_result = $auto_inc_check->fetch_assoc();
                $has_auto_inc = isset($auto_inc_result['EXTRA']) && strpos($auto_inc_result['EXTRA'], 'auto_increment') !== false;
                if (!$has_auto_inc) {
                    // Add AUTO_INCREMENT to requisition.id
                    if (!$conn->query("ALTER TABLE requisition MODIFY COLUMN id INT(11) NOT NULL AUTO_INCREMENT")) {
                        error_log("Warning: Failed to add AUTO_INCREMENT to requisition.id: " . $conn->error);
                        throw new Exception("Échec de la modification du schéma de la base de données. Veuillez contacter l'administrateur.");
                    }
                }
            }
            
            $check_auto_inc_item_sql = "SELECT COLUMN_NAME, EXTRA FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'requisition_item' AND COLUMN_NAME = 'id'";
            $auto_inc_item_check = $conn->query($check_auto_inc_item_sql);
            if ($auto_inc_item_check) {
                $auto_inc_item_result = $auto_inc_item_check->fetch_assoc();
                $has_auto_inc_item = isset($auto_inc_item_result['EXTRA']) && strpos($auto_inc_item_result['EXTRA'], 'auto_increment') !== false;
                if (!$has_auto_inc_item) {
                    // Add AUTO_INCREMENT to requisition_item.id
                    if (!$conn->query("ALTER TABLE requisition_item MODIFY COLUMN id INT(11) NOT NULL AUTO_INCREMENT")) {
                        error_log("Warning: Failed to add AUTO_INCREMENT to requisition_item.id: " . $conn->error);
                        throw new Exception("Échec de la modification du schéma de la base de données. Veuillez contacter l'administrateur.");
                    }
                }
            }
            
            // Ensure the database schema allows NULL for recipient_id when external
            // This is a one-time migration that will modify the schema if needed
            // NOTE: This must be done BEFORE beginning the transaction, as ALTER TABLE causes implicit commits
            if ($recipient_type === 'external') {
                // Check if recipient_id column allows NULL
                $check_null_sql = "SELECT IS_NULLABLE FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'requisition' AND COLUMN_NAME = 'recipient_id'";
                $null_check = $conn->query($check_null_sql);
                if ($null_check) {
                    $null_result = $null_check->fetch_assoc();
                    $allows_null = isset($null_result['IS_NULLABLE']) && $null_result['IS_NULLABLE'] === 'YES';
                    if (!$allows_null) {
                        // Modify the column to allow NULL and update foreign key constraint
                        // Drop the foreign key constraint first (if it exists)
                        $fk_check = $conn->query("SELECT CONSTRAINT_NAME FROM INFORMATION_SCHEMA.KEY_COLUMN_USAGE WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'requisition' AND CONSTRAINT_NAME = 'requisition_ibfk_5'");
                        if ($fk_check && $fk_check->num_rows > 0) {
                            if (!$conn->query("ALTER TABLE requisition DROP FOREIGN KEY requisition_ibfk_5")) {
                                error_log("Warning: Failed to drop foreign key constraint: " . $conn->error);
                            }
                        }
                        // Modify column to allow NULL
                        if (!$conn->query("ALTER TABLE requisition MODIFY COLUMN recipient_id INT(11) NULL DEFAULT NULL")) {
                            error_log("Warning: Failed to modify recipient_id column to allow NULL: " . $conn->error);
                            throw new Exception("Échec de la modification du schéma de la base de données. Veuillez contacter l'administrateur.");
                        }
                        // Re-add foreign key constraint (foreign keys can reference NULL values)
                        if (!$conn->query("ALTER TABLE requisition ADD CONSTRAINT requisition_ibfk_5 FOREIGN KEY (recipient_id) REFERENCES user(id) ON DELETE SET NULL")) {
                            error_log("Warning: Failed to re-add foreign key constraint: " . $conn->error);
                            // This is not critical, so we continue
                        }
                    }
                }
            }
            
            // Final validation: ensure province_id is valid before starting transaction
            if (empty($province_id) || !is_numeric($province_id) || $province_id <= 0) {
                throw new Exception("Erreur: Votre province n'est pas définie. Veuillez contacter l'administrateur.");
            }
            $province_id = (int)$province_id; // Ensure it's an integer
            
            $conn->begin_transaction();
            try {
                // Prepare the insert statement
                $statusToSave = $autoApproveRequisition ? 'approved' : 'pending';
                $approvedByToSave = $autoApproveRequisition ? $user_id : null;
                $originalAmountToSave = $total_amount;

                if ($recipient_type === 'external') {
                    // For external recipients, recipient_id should be NULL
                    $stmt = $conn->prepare("INSERT INTO requisition (user_id, total_amount, original_total_amount, status, approved_by, province_id, assigned_cashier_id, recipient_id, recipient_type, recipient_external_name, recipient_external_phone) VALUES (?, ?, ?, ?, ?, ?, ?, NULL, ?, ?, ?)");
                    if (!$stmt) {
                        throw new Exception("Query preparation failed for requisition: " . $conn->error . " (External recipient)");
                    }
                    // Parameters: user_id, total_amount, original_total_amount, status, approved_by, province_id, assigned_cashier_id, recipient_type, external_recipient_name, external_recipient_phone
                    $stmt->bind_param(
                        "iddsiiisss",
                        $user_id,                  // i - user_id
                        $total_amount,             // d - total_amount
                        $originalAmountToSave,     // d - original_total_amount
                        $statusToSave,             // s - status
                        $approvedByToSave,         // i - approved_by (nullable)
                        $province_id,              // i - province_id
                        $assigned_cashier_to_save, // i - assigned_cashier_id
                        $recipient_type,           // s - recipient_type (string)
                        $external_recipient_name,  // s - recipient_external_name
                        $external_recipient_phone  // s - recipient_external_phone
                    );
                } else {
                    // For internal recipients, use the actual recipient_id
                    // SQL binds: user_id, total_amount, original_total_amount, status, approved_by, province_id, assigned_cashier_id, recipient_id, recipient_type, external_recipient_name, external_recipient_phone
                    $stmt = $conn->prepare("INSERT INTO requisition (user_id, total_amount, original_total_amount, status, approved_by, province_id, assigned_cashier_id, recipient_id, recipient_type, recipient_external_name, recipient_external_phone) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
                    if (!$stmt) {
                        throw new Exception("Query preparation failed for requisition: " . $conn->error . " (Internal recipient)");
                    }
                    // For internal recipients, external fields should be empty strings
                    $external_name_internal = '';
                    $external_phone_internal = '';
                    $stmt->bind_param(
                        "iddsiiiisss",
                        $user_id,                  // i - user_id
                        $total_amount,             // d - total_amount
                        $originalAmountToSave,     // d - original_total_amount
                        $statusToSave,             // s - status
                        $approvedByToSave,         // i - approved_by (nullable)
                        $province_id,              // i - province_id
                        $assigned_cashier_to_save, // i - assigned_cashier_id
                        $recipient_id,             // i - recipient_id
                        $recipient_type,           // s - recipient_type
                        $external_name_internal,   // s - recipient_external_name (empty for internal)
                        $external_phone_internal    // s - recipient_external_phone (empty for internal)
                    );
                }
                
                if (!$stmt->execute()) {
                    throw new Exception("Échec de l'insertion de la réquisition: " . $stmt->error);
                }
                $requisition_id = $conn->insert_id;
                $stmt->close();

                $stmt = $conn->prepare("INSERT INTO requisition_item (requisition_id, category, description, quantity, unit_price) VALUES (?, ?, ?, ?, ?)");
                if (!$stmt) {
                    throw new Exception("Échec de la préparation de la requête pour requisition_item: " . $conn->error);
                }
                foreach ($valid_items as $item) {
                    $stmt->bind_param("issid", $requisition_id, $item['category'], $item['description'], $item['quantity'], $item['unit_price']);
                    $stmt->execute();
                }
                $stmt->close();

                $conn->commit();
                $recipient_descriptor = $recipient_type === 'external'
                    ? "external recipient '{$external_recipient_name}'"
                    : "recipient_id=" . ($recipient_id_param ?? 'NULL');
                $logMessage = "Created requisition ID $requisition_id for $total_amount BIF, $recipient_descriptor";
                if ($autoApproveRequisition) {
                    $logMessage .= " (auto-approved by {$role_name})";
                }
                log_action($user_id, 'create_requisition', $logMessage);
                
                if ($autoApproveRequisition) {
                    // Notify province cashiers that the requisition is already approved and pending payment
                    try {
                        $cashierRoleId = null;
                        $roleStmt = $conn->prepare("SELECT id FROM role WHERE name = 'Caissier_Comptable' LIMIT 1");
                        if ($roleStmt) {
                            $roleStmt->execute();
                            $roleResult = $roleStmt->get_result();
                            if ($roleRow = $roleResult->fetch_assoc()) {
                                $cashierRoleId = (int)$roleRow['id'];
                            }
                            $roleStmt->close();
                        }
                        if ($cashierRoleId) {
                            $cashierStmt = $conn->prepare("SELECT id FROM user WHERE role_id = ? AND province_id = ?");
                            if ($cashierStmt) {
                                $cashierStmt->bind_param('ii', $cashierRoleId, $province_id);
                                $cashierStmt->execute();
                                $cashierResult = $cashierStmt->get_result();
                                $cashiers = [];
                                while ($row = $cashierResult->fetch_assoc()) {
                                    $cashiers[] = (int)$row['id'];
                                }
                                $cashierStmt->close();

                                if (!empty($cashiers)) {
                                    $message = sprintf(
                                        'Réquisition #%d approuvée par %s (%s) et en attente de paiement. Montant: %s BIF.',
                                        $requisition_id,
                                        $full_name,
                                        $province ?? 'N/A',
                                        number_format($total_amount, 0, '.', ',')
                                    );
                                    $link = '/masunzu_bar_hotel/modules/cashiers/pay_requisitions.php';
                                    $notifyStmt = $conn->prepare("INSERT INTO user_notification (user_id, message, link, created_at) VALUES (?, ?, ?, NOW())");
                                    if ($notifyStmt) {
                                        foreach ($cashiers as $cashierId) {
                                            $notifyStmt->bind_param('iss', $cashierId, $message, $link);
                                            $notifyStmt->execute();
                                        }
                                        $notifyStmt->close();
                                    }
                                }
                            }
                        }
                    } catch (Exception $e) {
                        error_log("⚠️ Failed to notify cashiers for auto-approved requisition #{$requisition_id}: " . $e->getMessage());
                    }
                } else {
                    // Notify DGA and DG users about the new requisition (pending approval)
                    $stmtApprovers = $conn->prepare("SELECT u.id FROM user u JOIN role r ON u.role_id = r.id WHERE r.name IN ('DGA', 'DG')");
                    if ($stmtApprovers) {
                        $stmtApprovers->execute();
                        $approversResult = $stmtApprovers->get_result();
                        
                        $provinceName = $province ?? 'Unknown';
                        $notificationMessage = sprintf(
                            'Nouvelle réquisition #%d créée par %s (%s) pour %s BIF. Veuillez examiner et approuver.',
                            $requisition_id,
                            $full_name,
                            $provinceName,
                            number_format($total_amount, 0, ',', '.')
                        );
                        $notificationLink = '/masunzu_bar_hotel/modules/admin/approve_expenses.php';
                        
                        while ($approverRow = $approversResult->fetch_assoc()) {
                            $approverId = (int)$approverRow['id'];
                            $stmtNotification = $conn->prepare('INSERT INTO user_notification (user_id, message, link, created_at) VALUES (?, ?, ?, NOW())');
                            if ($stmtNotification) {
                                $stmtNotification->bind_param('iss', $approverId, $notificationMessage, $notificationLink);
                                if (!$stmtNotification->execute()) {
                                    error_log('Failed to send requisition notification to approver (DGA/DG): ' . $stmtNotification->error);
                                }
                                $stmtNotification->close();
                            }
                        }
                        $stmtApprovers->close();
                    }
                }
                
                header("Location: create.php?success=" . urlencode("Réquisition soumise avec succès."));
                exit;
            } catch (Exception $e) {
                $conn->rollback();
                $errors[] = "Échec de la soumission: " . $e->getMessage();
                error_log("ERROR: Failed to submit: " . $e->getMessage());
            }
        }
    }

    if ($role_name === 'Accountant' && $_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['process_payment'])) {
        error_log("DEBUG: Processing payment at " . date('Y-m-d H:i:s') . ", raw_post=" . json_encode($_POST));
        $requisition_id = filter_input(INPUT_POST, 'requisition_id', FILTER_VALIDATE_INT);
        $initiator_user_id = 0;
        $cashier_id = filter_input(INPUT_POST, 'cashier_id', FILTER_VALIDATE_INT);
        $receipt_number = trim(filter_input(INPUT_POST, 'receipt_number', FILTER_UNSAFE_RAW) ?? '');
        $recipient_signature = trim(filter_input(INPUT_POST, 'recipient_signature', FILTER_UNSAFE_RAW) ?? '');

        if (!$requisition_id) $errors[] = "ID de réquisition invalide.";
        if (!$cashier_id) $errors[] = "Sélectionnez un caissier.";
        if (empty($receipt_number)) $errors[] = "Entrez un numéro de reçu.";
        if (empty($recipient_signature)) $errors[] = "Entrez une signature du destinataire.";

        if (empty($errors)) {
            $stmt = $conn->prepare("SELECT total_amount, status, recipient_id, user_id AS initiator_user_id FROM requisition WHERE id = ? AND status = 'approved'");
            if (!$stmt) {
                throw new Exception("Échec de la préparation de la requête pour la vérification de la réquisition: " . $conn->error);
            }
            $stmt->bind_param("i", $requisition_id);
            $stmt->execute();
            $requisition = $stmt->get_result()->fetch_assoc();
            $stmt->close();

            if (!$requisition) {
                $errors[] = "Requisition not found or not approved.";
            } else {
                $initiator_user_id = (int)($requisition['initiator_user_id'] ?? 0);
                $total_amount = (float)$requisition['total_amount'];
                $conn->begin_transaction();
                $payment_successful = false;
                try {
                    $stmt = $conn->prepare("SELECT balance FROM accountant_balance WHERE accountant_id = ? FOR UPDATE");
                    if (!$stmt) {
                        throw new Exception("Query preparation failed for balance: " . $conn->error);
                    }
                    $stmt->bind_param("i", $user_id);
                    $stmt->execute();
                    $balance_row = $stmt->get_result()->fetch_assoc();
                    $stmt->close();
                    $accountant_balance = (float)($balance_row['balance'] ?? 0);
                    if ($accountant_balance < $total_amount) {
                        $conn->rollback();
                        $errors[] = "Insufficient funds: Available " . number_format($accountant_balance, 0, '.', ',') . " BIF, Required $total_amount BIF.";
                    } else {
                        $stmt = $conn->prepare("INSERT INTO expense (requisition_id, cashier_id, amount, receipt_number, recipient_signature, accountant_id, approval_status, payment_status) VALUES (?, ?, ?, ?, ?, ?, 'approved', 'processed')");
                        if (!$stmt) {
                            throw new Exception("Query preparation failed for expense: " . $conn->error);
                        }
                        $stmt->bind_param("iidssi", $requisition_id, $cashier_id, $total_amount, $receipt_number, $recipient_signature, $user_id);
                        $stmt->execute();
                        if ($stmt->affected_rows === 0) {
                            $stmt->close();
                            throw new Exception("Failed to record expense.");
                        }
                        $expense_id = (int)$conn->insert_id;
                        $stmt->close();
                        
                        // Enregistrer dans cashier_account si c'est un caissier (pas un comptable)
                        // Les réquisitions payées par caissier sont déjà gérées dans pay_requisitions.php
                        // Ici on gère les expenses créées directement (sans réquisition)
                        if ($cashier_id > 0 && empty($requisition_id)) {
                            // Ensure cashier_account table exists
                            $createCashierAccountSql = "
                                CREATE TABLE IF NOT EXISTS cashier_account (
                                    id INT AUTO_INCREMENT PRIMARY KEY,
                                    cashier_id INT NOT NULL,
                                    shift_id INT NULL,
                                    transaction_type ENUM('SHIFT_START','SHIFT_END','INVOICE_PAYMENT','LOAN_REPAYMENT','REQUISITION_PAYMENT','EXPENSE','BANK_DEPOSIT','CASH_TRANSFER','ADJUSTMENT') NOT NULL,
                                    direction ENUM('CREDIT','DEBIT') NOT NULL,
                                    amount BIGINT(20) NOT NULL,
                                    reference_type ENUM('SHIFT','INVOICE','LOAN_PAYMENT','REQUISITION','EXPENSE','BANK_DEPOSIT','CASH_TRANSFER','MANUAL') NOT NULL,
                                    reference_id INT NULL,
                                    description VARCHAR(255) DEFAULT NULL,
                                    created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
                                    created_by INT NULL,
                                    INDEX idx_cashier_created (cashier_id, created_at),
                                    INDEX idx_shift (shift_id),
                                    INDEX idx_reference (reference_type, reference_id),
                                    CONSTRAINT fk_ca_cashier FOREIGN KEY (cashier_id) REFERENCES user(id) ON DELETE CASCADE,
                                    CONSTRAINT fk_ca_shift FOREIGN KEY (shift_id) REFERENCES cashier_balance(id) ON DELETE SET NULL,
                                    CONSTRAINT fk_ca_creator FOREIGN KEY (created_by) REFERENCES user(id) ON DELETE SET NULL
                                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci
                            ";
                            @$conn->query($createCashierAccountSql);
                            
                            // Get shift_id
                            $stmtShift = $conn->prepare("SELECT id FROM cashier_balance WHERE cashier_id = ? AND shift_end IS NULL LIMIT 1");
                            $stmtShift->bind_param("i", $cashier_id);
                            $stmtShift->execute();
                            $shiftResult = $stmtShift->get_result();
                            $shiftRow = $shiftResult ? $shiftResult->fetch_assoc() : null;
                            $shift_id = $shiftRow ? (int)$shiftRow['id'] : null;
                            $stmtShift->close();
                            
                            if ($shift_id) {
                                $description = "Dépense: {$receipt_number}";
                                $stmtAccount = $conn->prepare("
                                    INSERT INTO cashier_account (cashier_id, shift_id, transaction_type, direction, amount, reference_type, reference_id, created_by, description)
                                    VALUES (?, ?, 'EXPENSE', 'DEBIT', ?, 'EXPENSE', ?, ?, ?)
                                ");
                                if ($stmtAccount) {
                                    $stmtAccount->bind_param("iiiiiis", $cashier_id, $shift_id, $total_amount, $expense_id, $user_id, $description);
                                    if (!$stmtAccount->execute()) {
                                        error_log("Failed to record EXPENSE in cashier_account: " . $stmtAccount->error);
                                    }
                                    $stmtAccount->close();
                                }
                            }
                        }

                        $stmt = $conn->prepare("UPDATE accountant_balance SET balance = balance - ?, last_updated = NOW() WHERE accountant_id = ?");
                        if (!$stmt) {
                            throw new Exception("Query preparation failed for balance update: " . $conn->error);
                        }
                        $stmt->bind_param("di", $total_amount, $user_id);
                        $stmt->execute();
                        if ($stmt->affected_rows === 0) {
                            $stmt->close();
                            throw new Exception("Failed to update accountant balance.");
                        }
                        $stmt->close();

                        $stmt = $conn->prepare("UPDATE requisition SET status = 'paid' WHERE id = ?");
                        if (!$stmt) {
                            throw new Exception("Query preparation failed for requisition update: " . $conn->error);
                        }
                        $stmt->bind_param("i", $requisition_id);
                        $stmt->execute();
                        if ($stmt->affected_rows === 0) {
                            $stmt->close();
                            throw new Exception("Failed to mark requisition as paid.");
                        }
                        $stmt->close();

                        $conn->commit();
                        $payment_successful = true;
                    }
                } catch (Exception $e) {
                    $conn->rollback();
                    if (strpos($e->getMessage(), 'Insufficient funds') === false) {
                        $errors[] = "Failed to process payment: " . $e->getMessage();
                        error_log("ERROR: Failed to process payment: " . $e->getMessage());
                    }
                }

                if ($payment_successful) {
                    $success = "Paiement traité avec succès.";
                    if ($initiator_user_id > 0) {
                        $paymentMessage = sprintf(
                            'Votre réquisition #%d de %s BIF a été payée.',
                            $requisition_id,
                            number_format($total_amount, 0, ',', '.')
                        );
                        $paymentLink = '/masunzu_bar_hotel/modules/expenses/expenses_history.php';
                        if (!mbh_notify_user($conn, $initiator_user_id, $paymentMessage, $paymentLink)) {
                            error_log("⚠️ Failed to notify initiator user_id {$initiator_user_id} about payment for requisition #{$requisition_id}");
                        }
                    }
                    log_action($user_id, 'process_expense_payment', "Processed payment for requisition ID $requisition_id, amount: $total_amount BIF, recipient_id: {$requisition['recipient_id']}");
                }
            }
        }
    }

    $requisitions = [];
    if ($role_name === 'Accountant') {
        $stmt = $conn->prepare("SELECT r.id,
                                       r.total_amount,
                                       r.status,
                                       r.created_at,
                                       u.full_name AS initiator_name,
                                       r.recipient_id,
                                       r.recipient_type,
                                       r.recipient_external_name,
                                       r.recipient_external_phone,
                                       u2.full_name AS recipient_name
                                FROM requisition r 
                                LEFT JOIN user u ON r.user_id = u.id 
                                LEFT JOIN user u2 ON r.recipient_id = u2.id 
                                WHERE r.province_id = ? AND r.status = 'approved' 
                                ORDER BY r.created_at DESC");
        if (!$stmt) {
            throw new Exception("Query preparation failed for requisitions: " . $conn->error);
        }
        $stmt->bind_param("i", $province_id);
        $stmt->execute();
        $requisitions = [];
        $result = $stmt->get_result();
        if ($result) {
            while ($row = $result->fetch_assoc()) {
                $recipientType = strtolower($row['recipient_type'] ?? 'internal');
                $isExternal = $recipientType === 'external';
                $externalName = trim((string)($row['recipient_external_name'] ?? ''));
                $externalPhone = trim((string)($row['recipient_external_phone'] ?? ''));
                $internalName = trim((string)($row['recipient_name'] ?? ''));
                    if ($isExternal) {
                    $baseName = $externalName !== '' ? $externalName : 'Non spécifié';
                    $row['recipient_display'] = 'Externe: ' . $baseName . ($externalPhone !== '' ? ' (' . $externalPhone . ')' : '');
                } else {
                    $baseName = $internalName !== '' ? $internalName : 'Non spécifié';
                    $row['recipient_display'] = 'Interne: ' . $baseName;
                }
                $requisitions[] = $row;
            }
            $result->free();
        }
        $stmt->close();

        $stmt = $conn->prepare("SELECT id, full_name FROM user WHERE role_id = (SELECT id FROM role WHERE name = 'Caissier_Comptable') AND province_id = ?");
        if (!$stmt) {
            throw new Exception("Query preparation failed for cashiers: " . $conn->error);
        }
        $stmt->bind_param("i", $province_id);
        $stmt->execute();
        $cashiers = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
        $stmt->close();
    }

    $recent_user_requisitions = [];
    if ($user_id) {
        $recentStmt = $conn->prepare("
            SELECT 
                r.id,
                r.total_amount,
                r.created_at,
                r.status AS approval_status,
                COALESCE(latest_expense.payment_status, 'unpaid') AS payment_status,
                latest_expense.approval_status AS expense_approval_status
            FROM requisition r
            LEFT JOIN (
                SELECT e1.requisition_id, e1.payment_status, e1.approval_status
                FROM expense e1
                JOIN (
                    SELECT requisition_id, MAX(id) AS max_id
                    FROM expense
                    GROUP BY requisition_id
                ) e2 ON e1.requisition_id = e2.requisition_id AND e1.id = e2.max_id
            ) latest_expense ON latest_expense.requisition_id = r.id
            WHERE r.user_id = ?
            ORDER BY r.created_at DESC
            LIMIT 5
        ");
        if ($recentStmt) {
            $recentStmt->bind_param("i", $user_id);
            $recentStmt->execute();
            $recent_user_requisitions = $recentStmt->get_result()->fetch_all(MYSQLI_ASSOC);
            $recentStmt->close();
        } else {
            error_log("ERROR: Failed to prepare recent requisitions query: " . $conn->error);
        }
    }

    $conn->close();

} catch (Exception $e) {

} catch (Exception $e) {
    error_log("ERROR in expenses/create.php: " . $e->getMessage());
    echo "<p style='color: #FF0000; text-align: center; padding: 20px;'>Erreur: " . htmlspecialchars($e->getMessage()) . "</p>";
    include $footer_path ?: "</body></html>";
    exit;
}

include $header_path ?: die("Error: Header file missing. Contact administrator.");
?>
<style>
    .expenses-page {
        background: linear-gradient(180deg, #f7f2eb 0%, #fdf4e8 40%, #f7f2eb 100%);
        padding: 60px 20px 80px;
        min-height: 100vh;
    }
    .expenses-card {
        max-width: 920px;
        margin: 0 auto;
        background: #ffffff;
        border-radius: 28px;
        padding: 32px 36px;
        box-shadow: 0 24px 45px rgba(0, 0, 0, 0.08);
        border: 1px solid rgba(75, 47, 31, 0.08);
    }
    .page-heading h2 {
        margin: 0;
        font-size: 32px;
        color: #331e10;
    }
    .page-heading .page-intro {
        margin-top: 10px;
        color: #4b2f1f;
        font-size: 16px;
        line-height: 1.5;
    }
    .amount-card {
        background: linear-gradient(135deg, #f8b745 0%, #f4a261 100%);
        padding: 20px;
        border-radius: 16px;
        margin-bottom: 18px;
        border: 1px solid rgba(255, 255, 255, 0.4);
        box-shadow: inset 0 0 0 1px rgba(255, 255, 255, 0.4);
    }
    .amount-card .amount-label {
        margin: 0;
        font-size: 14px;
        text-transform: uppercase;
        letter-spacing: 0.15em;
        color: #4b2f1f;
    }
    .amount-card .amount-value {
        margin: 4px 0 0;
        font-size: 32px;
        font-weight: 700;
        color: #2f1b0f;
    }
    .create-form {
        display: flex;
        flex-direction: column;
        gap: 18px;
    }
    .form-section {
        background: #fdf9f4;
        border-radius: 16px;
        border: 1px solid #f1dfc5;
        padding: 20px;
        box-shadow: 0 12px 32px rgba(0, 0, 0, 0.05);
        display: flex;
        flex-direction: column;
        gap: 18px;
    }
    .form-section .section-heading {
        margin: 0;
        font-size: 20px;
        color: #2c1810;
    }
    .form-group {
        display: flex;
        flex-direction: column;
        gap: 12px;
    }
    .form-group label {
        font-weight: 600;
        color: #4b2f1f;
    }
    .form-group select,
    .form-group input,
    .form-group textarea {
        width: 100%;
        padding: 10px;
        border-radius: 8px;
        border: 1px solid #4b2f1f;
        font-size: 14px;
        font-family: inherit;
        background: #fff;
    }
    /* Hide spinner controls on number inputs */
    .form-group input[type="number"] {
        -moz-appearance: textfield;
    }
    .form-group input[type="number"]::-webkit-outer-spin-button,
    .form-group input[type="number"]::-webkit-inner-spin-button {
        -webkit-appearance: none;
        margin: 0;
    }
    .form-group textarea {
        min-height: 90px;
        resize: vertical;
    }
    .recipient-type-group {
        flex-direction: row;
        align-items: center;
        flex-wrap: wrap;
    }
    .recipient-type-group span {
        font-weight: 600;
        color: #4b2f1f;
    }
    .recipient-type-group label {
        display: flex;
        align-items: center;
        gap: 6px;
        color: #2c1810;
    }
    .province-info {
        color: #4b2f1f;
        font-size: 14px;
        margin: 0;
        font-weight: 600;
    }
    .form-actions {
        display: flex;
        flex-wrap: wrap;
        gap: 14px;
    }
    .form-actions button {
        flex: 1;
        min-width: 200px;
        max-width: 280px;
    }
    .expense-item-card {
        border: 1px solid #4b2f1f;
        border-radius: 12px;
        padding: 16px;
        background: #fffdf9;
        box-shadow: 0 12px 24px rgba(0, 0, 0, 0.08);
        display: flex;
        flex-direction: column;
        gap: 12px;
    }
    .expense-item-card h4 {
        margin: 0;
        color: #4b2f1f;
    }
    #items-container {
        display: flex;
        flex-direction: column;
        gap: 16px;
    }
    .recent-user-requisitions {
        margin-top: 30px;
        background: #fff;
        border-radius: 18px;
        border: 1px solid #ebe6dc;
        padding: 26px;
        box-shadow: 0 14px 32px rgba(0, 0, 0, 0.06);
    }
    .recent-user-requisitions h3 {
        margin: 0;
        color: #2c1810;
        font-size: 24px;
    }
    .status-pill {
        display: inline-flex;
        align-items: center;
        padding: 4px 12px;
        border-radius: 999px;
        font-size: 12px;
        font-weight: 600;
        color: #fff;
    }
    .status-pill.approved {
        background: #2e7d32;
    }
    .status-pill.pending {
        background: #f59e0b;
    }
    .status-pill.rejected {
        background: #c0262c;
    }
    .status-pill.unpaid {
        background: #9ca3af;
        color: #1f2937;
    }
    .status-pill.paid {
        background: #16a34a;
    }
    .recent-view-link {
        display: inline-flex;
        align-items: center;
        justify-content: center;
        padding: 6px 14px;
        border-radius: 8px;
        background: #4b2f1f;
        color: #f4f0e4;
        text-decoration: none;
        font-size: 13px;
        font-weight: 600;
    }
    @media (max-width: 900px) {
        .expenses-card {
            padding: 24px;
        }
        .form-actions {
            flex-direction: column;
            align-items: stretch;
        }
        .form-actions button {
            max-width: none;
        }
    }
</style>
<div class="expenses-page">
    <div class="expenses-card">
        <div class="page-heading">
            <h2>Créer Réquisition Dépenses</h2>
            <p class="page-intro">Bienvenue, <?php echo htmlspecialchars($full_name); ?> (<?php echo htmlspecialchars($role_name); ?>, <?php echo htmlspecialchars($province); ?>)</p>
        </div>

        <?php if ($success): ?>
            <p style="color: #4CAF50; font-size: 16px; text-align: center;"><?php echo htmlspecialchars($success); ?></p>
        <?php endif; ?>
        <?php if (!empty($errors)): ?>
            <p style="color: #FF0000; font-size: 16px; text-align: center;">Veuillez corriger les erreurs suivantes:</p>
            <?php foreach ($errors as $error): ?>
                <p style="color: #FF0000; font-size: 16px; text-align: center;"><?php echo htmlspecialchars($error); ?></p>
            <?php endforeach; ?>
            <p style="color: #FF0000; font-size: 16px; text-align: center;">Données soumises: <?php echo htmlspecialchars(json_encode($_POST)); ?></p>
        <?php endif; ?>
        
        <div class="amount-card">
            <p class="amount-label">Montant Total:</p>
            <p class="amount-value">
                <span id="total-amount-display">0</span> BIF
            </p>
        </div>

        <form method="POST" id="requisition-form" class="create-form" action="create.php" onsubmit="return previewExpense(event)">
            <input type="hidden" name="submit_requisition" value="1">
            <section class="form-section recipient-section">
                <h3 class="section-heading">Nouvelle Réquisition Dépenses</h3>
            <?php if (empty($province_id) && !empty($provinceOptions)): ?>
                <div class="form-group">
                    <label for="province_id">Province assignée <span style="color: #FF0000;">*</span></label>
                    <select name="province_id" id="province_id" required>
                            <option value="">Sélectionner une province</option>
                            <?php foreach ($provinceOptions as $provOption): ?>
                                <option value="<?php echo htmlspecialchars($provOption['id']); ?>" <?php echo ((int)$province_id === $provOption['id']) ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($provOption['name']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                        <p style="color: #666; font-size: 12px; margin-top: 5px;">Veuillez sélectionner une province pour continuer.</p>
                    </div>
                <?php else: ?>
                    <p class="province-info">Province actuelle: <?php echo htmlspecialchars($province); ?></p>
                <?php endif; ?>
                <?php if ($requires_requisition_cashier): ?>
                    <div class="form-group">
                        <label for="assigned_cashier_id">Caissier assigné</label>
                        <select name="assigned_cashier_id" id="assigned_cashier_id" required>
                            <option value="">Sélectionner un caissier</option>
                            <?php foreach ($cashiers as $cashier): ?>
                                <option value="<?php echo (int)$cashier['id']; ?>" <?php echo ($selected_assigned_cashier_id === (int)$cashier['id']) ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($cashier['full_name'], ENT_QUOTES, 'UTF-8'); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                <?php else: ?>
                    <input type="hidden" name="assigned_cashier_id" value="<?php echo (int)$assigned_cashier_to_save; ?>">
                <?php endif; ?>
                <div class="form-group recipient-type-group">
                    <span>Type de Destinataire:</span>
                    <label>
                        <input type="radio" name="recipient_type" value="internal" checked> Personnel interne
                    </label>
                    <label>
                        <input type="radio" name="recipient_type" value="external"> Personne externe
                    </label>
                </div>
                <div id="internal-recipient-container" class="form-group">
                    <label for="recipient_id">Destinataire (Receveur d'Espèces):</label>
                    <select name="recipient_id" id="recipient_id">
                        <option value="">-- Choisissez un personnel interne --</option>
                        <?php 
                        if (!empty($users)): 
                            foreach ($users as $user): 
                                if (isset($user['id']) && isset($user['full_name'])):
                                    $optionValue = (int)$user['id'];
                                    $isSelectedOption = $optionValue === $selectedRecipientId;
                        ?>
                            <option value="<?php echo $optionValue; ?>" data-full-name="<?php echo htmlspecialchars($user['full_name'], ENT_QUOTES, 'UTF-8'); ?>" <?php echo $isSelectedOption ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($user['full_name']); ?>
                            </option>
                        <?php 
                                endif;
                            endforeach; 
                        endif; 
                        ?>
                    </select>
                    <p id="selected-internal-recipient" style="color: #4B2F1F; font-size: 13px; margin-top: 6px; display: none;">Nom du destinataire interne: <span id="selected-internal-recipient-name" style="font-weight: 600;">—</span></p>
                    <?php if (empty($users)): ?>
                        <p style="color: #FF0000; font-size: 12px; margin-top: 5px;">Aucun utilisateur trouvé dans la base de données (sauf vous).</p>
                    <?php endif; ?>
                </div>
                <div id="external-recipient-container" class="form-group" style="display: none;">
                    <label for="external_recipient_name">Nom Destinataire Externe</label>
                    <input type="text" id="external_recipient_name" name="external_recipient_name" required>
                    <label for="external_recipient_phone">Téléphone Destinataire Externe</label>
                    <input type="text" id="external_recipient_phone" name="external_recipient_phone" placeholder="ex: 788123456" required pattern="[0-9]{8,15}" title="8-15 chiffres requis">
                </div>
            </section>
            <section class="form-section items-section">
                <h3 class="section-heading">Articles de Dépense</h3>
                <div id="items-container">
                    <div class="expense-item expense-item-card">
                        <h4>Article 1</h4>
                        <div class="form-group">
                            <label for="items_0_category">Catégorie:</label>
                            <select name="items[0][category]" id="items_0_category" required onchange="updateTotalAmount()">
                                <option value="">Sélectionner Catégorie</option>
                                <?php 
                                    $category_labels = [
                                        'communication' => 'Communication',
                                        'transport' => 'Transport',
                                        'frais_de_mission' => 'Frais de Mission',
                                        'office_utilities' => 'Utilitaires Bureau',
                                        'maintenance' => 'Maintenance',
                                        'others' => 'Autres'
                                    ];
                                    foreach ($categories as $category): 
                                        $label = isset($category_labels[$category]) ? $category_labels[$category] : ucwords(str_replace('_', ' ', $category));
                                ?>
                                    <option value="<?php echo htmlspecialchars($category); ?>"><?php echo htmlspecialchars($label); ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="form-group">
                            <label for="items_0_description">Description:</label>
                            <textarea name="items[0][description]" id="items_0_description" required></textarea>
                        </div>
                        <div class="form-group">
                            <label for="items_0_quantity">Quantité:</label>
                            <input type="number" name="items[0][quantity]" id="items_0_quantity" min="1" required oninput="this.value = this.value.replace(/[^0-9]/g, ''); updateTotalAmount();">
                        </div>
                        <div class="form-group">
                            <label for="items_0_unit_price">Prix Unitaire (BIF):</label>
                            <input type="number" name="items[0][unit_price]" id="items_0_unit_price" step="0.01" min="0.01" required oninput="this.value = this.value.replace(/[^0-9.]/g, ''); updateTotalAmount();">
                        </div>
                    </div>
                </div>
            </section>
            <div class="form-actions">
                <button type="button" onclick="addItem()" style="background-color: #4B2F1F; color: #F4F0E4; padding: 12px 18px; border: none; border-radius: 5px; cursor: pointer;">Ajouter Un Autre Article</button>
                <button type="submit" id="submit-button" style="background: linear-gradient(135deg, #4B2F1F 0%, #6B4F2F 100%); color: #F4F0E4; padding: 12px 30px; border: none; border-radius: 5px; cursor: pointer; font-weight: bold; font-size: 16px; transition: all 0.3s ease;">Prévisualiser & Soumettre</button>
            </div>
        </form>

        <section class="recent-user-requisitions">
            <h3>Vos dernières réquisitions</h3>
            <?php 
                $approvalLabels = [
                    'approved' => 'Approuvée',
                    'pending' => 'En attente',
                    'rejected' => 'Rejetée',
                ];
                $paymentLabels = [
                    'processed' => 'Payée',
                    'pending' => 'Paiement en attente',
                    'unpaid' => 'Non payée',
                ];
            ?>
            <?php if (!empty($recent_user_requisitions)): ?>
                <div class="recent-table-wrapper">
                    <table class="recent-requisition-table">
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>Date</th>
                                <th>Montant (BIF)</th>
                                <th>Approbation</th>
                                <th>Paiement</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($recent_user_requisitions as $req): ?>
                                <?php
                                    $approvalStatus = strtolower($req['approval_status'] ?? 'pending');
                                    $paymentStatus = strtolower($req['payment_status'] ?? 'unpaid');
                                    $approvalLabel = $approvalLabels[$approvalStatus] ?? ucfirst($approvalStatus);
                                    $paymentLabel = $paymentLabels[$paymentStatus] ?? ucfirst($paymentStatus);
                                    $approvalClass = $approvalStatus === 'approved' ? 'approved' : ($approvalStatus === 'rejected' ? 'rejected' : 'pending');
                                    $paymentClass = $paymentStatus === 'processed' ? 'paid' : ($paymentStatus === 'unpaid' ? 'unpaid' : 'pending');
                                ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($req['id']); ?></td>
                                    <td><?php echo htmlspecialchars(date('d/m/Y H:i', strtotime($req['created_at'] ?? 'now'))); ?></td>
                                    <td><?php echo number_format((float)$req['total_amount'], 0, ',', '.'); ?></td>
                                    <td><span class="status-pill <?php echo $approvalClass; ?>"><?php echo htmlspecialchars($approvalLabel); ?></span></td>
                                    <td><span class="status-pill <?php echo $paymentClass; ?>"><?php echo htmlspecialchars($paymentLabel); ?></span></td>
                                    <td>
                                        <a class="recent-view-link" href="/masunzu_bar_hotel/modules/expenses/expenses_history.php?search=<?php echo urlencode($req['id']); ?>">Voir</a>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php else: ?>
                <p style="color: #4B2F1F; font-size: 14px; margin-top: 8px;">Aucune réquisition récente pour le moment.</p>
            <?php endif; ?>
        </section>

        <!-- Preview Modal -->
        <div id="preview-modal" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.5); backdrop-filter: blur(4px); z-index: 1000; animation: fadeIn 0.3s ease;">
            <div style="position: fixed; top: 50%; left: 50%; transform: translate(-50%, -50%); background: #F4F0E4; border-radius: 12px; box-shadow: 0 20px 60px rgba(0,0,0,0.3); max-width: 600px; width: 90%; max-height: 90vh; overflow-y: auto; animation: slideDown 0.3s ease; z-index: 1001;">
                <!-- Modal Header -->
                <div style="background: linear-gradient(135deg, #4B2F1F 0%, #F4A261 100%); padding: 20px; border-radius: 12px 12px 0 0; display: flex; justify-content: space-between; align-items: center;">
                    <h2 style="color: #F4F0E4; margin: 0; font-size: 24px;">Prévisualisation Dépense</h2>
                    <button onclick="closePreviewModal()" style="background: none; border: none; color: #F4F0E4; font-size: 28px; cursor: pointer; padding: 0; width: 30px; height: 30px; display: flex; align-items: center; justify-content: center;">×</button>
                </div>

                <!-- Modal Content -->
                <div style="padding: 25px;">
                    <!-- Recipient Card -->
                    <div style="background: #FFFFFF; padding: 15px; border-radius: 8px; margin-bottom: 15px; border-left: 4px solid #F4A261;">
                        <p style="color: #999; font-size: 12px; margin: 0 0 5px 0; text-transform: uppercase;">Destinataire</p>
                        <p style="color: #4B2F1F; font-size: 16px; font-weight: bold; margin: 0;" id="preview-recipient"></p>
                    </div>

                    <!-- Total Amount Card -->
                    <div style="background: linear-gradient(135deg, #F4A261 0%, #F4B580 100%); padding: 15px; border-radius: 8px; margin-bottom: 15px;">
                        <p style="color: #4B2F1F; font-size: 12px; margin: 0 0 5px 0; text-transform: uppercase;">Montant Total</p>
                        <p style="color: #4B2F1F; font-size: 24px; font-weight: bold; margin: 0;" id="preview-total-amount"></p>
                    </div>

                    <!-- Items Section -->
                    <div style="margin-bottom: 20px;">
                        <h3 style="color: #4B2F1F; margin: 0 0 12px 0; font-size: 16px;">Articles de Dépense</h3>
                        <div id="preview-items-container"></div>
                    </div>

                    <!-- Action Buttons -->
                    <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 10px; margin-top: 20px;">
                        <button onclick="closePreviewModal()" style="background: #E0E0E0; color: #4B2F1F; padding: 12px 20px; border: none; border-radius: 6px; cursor: pointer; font-weight: bold; font-size: 14px; transition: all 0.3s ease;">Annuler</button>
                        <button onclick="submitExpenseForm()" style="background: linear-gradient(135deg, #4CAF50 0%, #66BB6A 100%); color: white; padding: 12px 20px; border: none; border-radius: 6px; cursor: pointer; font-weight: bold; font-size: 14px; transition: all 0.3s ease;">Confirmer & Soumettre</button>
                    </div>
                </div>
            </div>
        </div>

        <style>
            @keyframes fadeIn {
                from { opacity: 0; }
                to { opacity: 1; }
            }
            @keyframes slideDown {
                from {
                    opacity: 0;
                    transform: translate(-50%, -55%);
                }
                to {
                    opacity: 1;
                    transform: translate(-50%, -50%);
                }
            }
            #preview-modal.show {
                display: block !important;
            }
        .item-card {
            background: #FFFFFF;
            padding: 12px;
            border-radius: 6px;
            margin-bottom: 10px;
            border-left: 4px solid #F4A261;
        }
        .item-card p {
            margin: 5px 0;
            font-size: 13px;
            color: #666;
        }
        .item-card .category {
            color: #4B2F1F;
            font-weight: bold;
            text-transform: uppercase;
            font-size: 12px;
            margin-bottom: 8px;
        }
        .recent-user-requisitions {
            margin-top: 30px;
            background: #fff;
            border-radius: 14px;
            border: 1px solid #E4E0D5;
            padding: 24px;
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.08);
        }
        .recent-user-requisitions h3 {
            margin-top: 0;
            color: #2C1810;
            font-size: 22px;
        }
        .recent-requisition-table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 15px;
        }
        .recent-requisition-table th,
        .recent-requisition-table td {
            padding: 10px 8px;
            border-bottom: 1px solid #F0E7DC;
            font-size: 14px;
            text-align: left;
        }
        .recent-requisition-table thead {
            background: #4B2F1F;
            color: #F4F0E4;
        }
        .recent-requisition-table tbody tr:nth-child(odd) {
            background: #FAF5EF;
        }
        .status-pill {
            display: inline-flex;
            align-items: center;
            padding: 4px 10px;
            border-radius: 999px;
            font-size: 12px;
            font-weight: 600;
            color: #fff;
        }
        .status-pill.approved {
            background: #2E7D32;
        }
        .status-pill.pending {
            background: #F59E0B;
        }
        .status-pill.rejected {
            background: #C0262C;
        }
        .status-pill.unpaid {
            background: #9CA3AF;
            color: #1F2937;
        }
        .status-pill.paid {
            background: #16A34A;
        }
        .recent-view-link {
            display: inline-flex;
            align-items: center;
            justify-content: center;
            padding: 6px 12px;
            border-radius: 8px;
            background: #4B2F1F;
            color: #F4F0E4;
            text-decoration: none;
            font-size: 13px;
            font-weight: 600;
            transition: transform 0.2s ease, box-shadow 0.2s ease;
        }
        .recent-view-link:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 16px rgba(0, 0, 0, 0.15);
        }
        </style>

        <script>
            let itemCount = 1;
            let formSubmitting = false;
            let skipPreview = false;

            function updateTotalAmount() {
                const items = document.querySelectorAll('.expense-item');
                let total = 0;
                
                items.forEach((item, index) => {
                    const quantity = item.querySelector(`input[name="items[${index}][quantity]"]`).value.trim();
                    const unit_price = item.querySelector(`input[name="items[${index}][unit_price]"]`).value.trim();
                    
                    if (quantity && unit_price && !isNaN(quantity) && !isNaN(unit_price)) {
                        total += parseInt(quantity) * parseFloat(unit_price);
                    }
                });
                
                const totalDisplay = document.getElementById('total-amount-display');
                totalDisplay.textContent = new Intl.NumberFormat('fr-FR').format(Math.round(total));
            }

            function addItem() {
                console.log('DEBUG: Adding item, count: ' + itemCount);
                const container = document.getElementById('items-container');
                const newItem = document.createElement('div');
                newItem.className = 'expense-item expense-item-card';
                newItem.innerHTML = `
                    <h4>Article ${itemCount + 1}</h4>
                    <div class="form-group">
                        <label for="items_${itemCount}_category">Catégorie:</label>
                        <select name="items[${itemCount}][category]" id="items_${itemCount}_category" required onchange="updateTotalAmount()">
                            <option value="">Sélectionner Catégorie</option>
                            <?php 
                                $category_labels = [
                                    'communication' => 'Communication',
                                    'transport' => 'Transport',
                                    'frais_de_mission' => 'Frais de Mission',
                                    'office_utilities' => 'Utilitaires Bureau',
                                    'maintenance' => 'Maintenance',
                                    'others' => 'Autres'
                                ];
                                foreach ($categories as $category): 
                                    $label = isset($category_labels[$category]) ? $category_labels[$category] : ucwords(str_replace('_', ' ', $category));
                            ?>
                                <option value="<?php echo htmlspecialchars($category); ?>"><?php echo htmlspecialchars($label); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="form-group">
                        <label for="items_${itemCount}_description">Description:</label>
                        <textarea name="items[${itemCount}][description]" id="items_${itemCount}_description" required></textarea>
                    </div>
                    <div class="form-group">
                        <label for="items_${itemCount}_quantity">Quantité:</label>
                        <input type="number" name="items[${itemCount}][quantity]" id="items_${itemCount}_quantity" min="1" required oninput="this.value = this.value.replace(/[^0-9]/g, ''); updateTotalAmount();">
                    </div>
                    <div class="form-group">
                        <label for="items_${itemCount}_unit_price">Prix Unitaire (BIF):</label>
                        <input type="number" name="items[${itemCount}][unit_price]" id="items_${itemCount}_unit_price" step="0.01" min="0.01" required oninput="this.value = this.value.replace(/[^0-9.]/g, ''); updateTotalAmount();">
                    </div>
                    <button type="button" onclick="this.parentElement.remove(); updateTotalAmount();" style="background-color: #FF0000; color: #FFFFFF; padding: 5px 10px; border: none; border-radius: 5px; cursor: pointer;">Supprimer</button>
                `;
                container.appendChild(newItem);
                itemCount++;
            }

            function validateForm() {
                console.log('DEBUG: Validating form at ' + new Date().toLocaleString());
                const recipientType = document.querySelector('input[name="recipient_type"]:checked')?.value || 'internal';
                const recipientId = document.getElementById('recipient_id')?.value || '';
                const externalNameEl = document.getElementById('external_recipient_name');
                const externalPhoneEl = document.getElementById('external_recipient_phone');
                const externalName = externalNameEl ? externalNameEl.value.trim() : '';
                const externalPhone = externalPhoneEl ? externalPhoneEl.value.trim() : '';
                const items = document.querySelectorAll('.expense-item');
                let valid = true;
                let errors = [];

                if (recipientType === 'internal') {
                    if (!recipientId) {
                        errors.push('Veuillez sélectionner un destinataire.');
                        valid = false;
                    }
                } else {
                    if (!externalName) {
                        errors.push("Veuillez entrer le nom du destinataire externe.");
                        valid = false;
                    }
                    if (!externalPhone || !externalPhone.match(/^[0-9]{8,15}$/)) {
                        errors.push('Veuillez entrer un numéro de téléphone valide (8-15 chiffres) pour le destinataire externe.');
                        valid = false;
                    }
                }

                items.forEach((item, index) => {
                    const category = item.querySelector(`select[name="items[${index}][category]"]`).value;
                    const description = item.querySelector(`textarea[name="items[${index}][description]"]`).value.trim();
                    const quantity = item.querySelector(`input[name="items[${index}][quantity]"]`).value.trim();
                    const unit_price = item.querySelector(`input[name="items[${index}][unit_price]"]`).value.trim();

                    console.log(`DEBUG: Item ${index + 1}: category=${category}, description=${description}, quantity=${quantity}, unit_price=${unit_price}`);

                    if (!category) {
                        errors.push(`Sélectionnez une catégorie pour l'article ${index + 1}.`);
                        valid = false;
                    }
                    if (!description || description.match(/^\s*$/)) {
                        errors.push(`Entrez une description valide pour l'article ${index + 1}.`);
                        valid = false;
                    }
                    if (!quantity || isNaN(quantity) || parseInt(quantity) <= 0) {
                        errors.push(`Entrez une quantité positive pour l'article ${index + 1}.`);
                        valid = false;
                    }
                    if (!unit_price || isNaN(unit_price) || parseFloat(unit_price) <= 0) {
                        errors.push(`Entrez un prix unitaire positif pour l'article ${index + 1}.`);
                        valid = false;
                    }
                });

                if (!valid) {
                    console.log('DEBUG: Validation errors: ' + errors.join('; '));
                    alert('Veuillez corriger les erreurs suivantes:\n' + errors.join('\n'));
                    return false;
                }

                console.log('DEBUG: Form validated successfully');
                return true;
            }

            function getInternalRecipientLabel() {
                const select = document.getElementById('recipient_id');
                if (!select) {
                    return '—';
                }
                const currentValue = select.value;
                const options = Array.from ? Array.from(select.options) : Array.prototype.slice.call(select.options);
                let selectedOption = options.find(option => option.value === currentValue);
                if (!selectedOption && select.selectedIndex >= 0) {
                    selectedOption = select.options[select.selectedIndex];
                }
                if (!selectedOption) {
                    return '—';
                }
                const label = selectedOption.dataset.fullName || selectedOption.textContent;
                return label ? label.trim() : '—';
            }

            function previewExpense(event) {
                // If we're submitting from the modal, skip preview
                if (skipPreview) {
                    console.log('DEBUG: Skipping preview, submitting directly');
                    skipPreview = false;
                    return true;
                }
                
                event.preventDefault();
                event.stopPropagation();
                
                console.log('DEBUG: previewExpense called at ' + new Date().toLocaleString());
                
                if (!validateForm()) {
                    console.log('DEBUG: Form validation failed');
                    return false;
                }

                console.log('DEBUG: Form validation passed, showing preview');
                
                // Get form data
                const recipientType = document.querySelector('input[name="recipient_type"]:checked')?.value || 'internal';
                let recipientText = '';
                if (recipientType === 'internal') {
                    recipientText = getInternalRecipientLabel();
                } else {
                    const externalName = document.getElementById('external_recipient_name').value.trim();
                    const externalPhone = document.getElementById('external_recipient_phone').value.trim();
                    recipientText = externalName + (externalPhone ? ` (${externalPhone})` : '');
                }
                const items = document.querySelectorAll('.expense-item');
                let total = 0;
                let itemsData = [];

                items.forEach((item, index) => {
                    const category = item.querySelector(`select[name="items[${index}][category]"]`).value;
                    const description = item.querySelector(`textarea[name="items[${index}][description]"]`).value.trim();
                    const quantity = parseInt(item.querySelector(`input[name="items[${index}][quantity]"]`).value);
                    const unit_price = parseFloat(item.querySelector(`input[name="items[${index}][unit_price]"]`).value);
                    const itemTotal = quantity * unit_price;
                    
                    total += itemTotal;
                    itemsData.push({
                        category: category,
                        description: description,
                        quantity: quantity,
                        unit_price: unit_price,
                        itemTotal: itemTotal
                    });
                });

                // Populate modal
                document.getElementById('preview-recipient').textContent = recipientText;
                document.getElementById('preview-total-amount').textContent = new Intl.NumberFormat('fr-FR').format(Math.round(total)) + ' BIF';

                // Build items preview
                const itemsContainer = document.getElementById('preview-items-container');
                itemsContainer.innerHTML = '';
                itemsData.forEach((item, index) => {
                    const itemCard = document.createElement('div');
                    itemCard.className = 'item-card';
                    const categoryLabels = {
                        'communication': 'Communication',
                        'transport': 'Transport',
                        'frais_de_mission': 'Frais de Mission',
                        'office_utilities': 'Utilitaires Bureau',
                        'maintenance': 'Maintenance',
                        'others': 'Autres'
                    };
                    const categoryLabel = categoryLabels[item.category] || item.category.replace(/_/g, ' ');
                    itemCard.innerHTML = `
                        <div class="category">${categoryLabel}</div>
                        <p><strong>Description:</strong> ${item.description}</p>
                        <p><strong>Quantité:</strong> ${item.quantity}</p>
                        <p><strong>Prix Unitaire:</strong> ${new Intl.NumberFormat('fr-FR').format(item.unit_price)} BIF</p>
                        <p style="background: #F4A261; padding: 5px 8px; border-radius: 4px; margin: 8px 0 0 0 !important; color: #4B2F1F; font-weight: bold;">Sous-total: ${new Intl.NumberFormat('fr-FR').format(Math.round(item.itemTotal))} BIF</p>
                    `;
                    itemsContainer.appendChild(itemCard);
                });

                // Show modal
                const modal = document.getElementById('preview-modal');
                modal.style.display = 'block';
                modal.classList.add('show');

                // Close modal when clicking outside
                modal.addEventListener('click', function(e) {
                    if (e.target === modal) {
                        closePreviewModal();
                    }
                });

                // Close modal with Escape key
                document.addEventListener('keydown', function(e) {
                    if (e.key === 'Escape' && modal.style.display === 'block') {
                        closePreviewModal();
                    }
                });

                return false;
            }

            function closePreviewModal() {
                const modal = document.getElementById('preview-modal');
                modal.style.display = 'none';
                modal.classList.remove('show');
            }

            function submitExpenseForm() {
                if (formSubmitting) {
                    console.log('DEBUG: Form already submitting, ignoring duplicate submit');
                    return;
                }
                console.log('DEBUG: submitExpenseForm called at ' + new Date().toLocaleString());
                
                // Close modal first
                closePreviewModal();
                
                // Set flag to skip preview on next submit
                skipPreview = true;
                formSubmitting = true;
                
                // Ensure required attributes are set correctly before submission
                const recipientType = document.querySelector('input[name="recipient_type"]:checked')?.value || 'internal';
                if (recipientType === 'external') {
                    const externalNameEl = document.getElementById('external_recipient_name');
                    const externalPhoneEl = document.getElementById('external_recipient_phone');
                    const recipientIdEl = document.getElementById('recipient_id');
                    if (externalNameEl) externalNameEl.setAttribute('required', 'required');
                    if (externalPhoneEl) externalPhoneEl.setAttribute('required', 'required');
                    if (recipientIdEl) recipientIdEl.removeAttribute('required');
                } else {
                    const recipientIdEl = document.getElementById('recipient_id');
                    const externalNameEl = document.getElementById('external_recipient_name');
                    const externalPhoneEl = document.getElementById('external_recipient_phone');
                    if (recipientIdEl) recipientIdEl.setAttribute('required', 'required');
                    if (externalNameEl) externalNameEl.removeAttribute('required');
                    if (externalPhoneEl) externalPhoneEl.removeAttribute('required');
                }
                
                // Use a small delay to ensure modal is closed and DOM is ready
                setTimeout(function() {
                    const form = document.getElementById('requisition-form');
                    if (form) {
                        console.log('DEBUG: Submitting form');
                        // Remove onsubmit temporarily to bypass preview
                        const originalOnsubmit = form.onsubmit;
                        form.onsubmit = null;
                        form.submit();
                        // Restore onsubmit (though form is submitting, this is just cleanup)
                        form.onsubmit = originalOnsubmit;
                    } else {
                        console.error('DEBUG: Form element not found!');
                        formSubmitting = false;
                        skipPreview = false;
                    }
                }, 100);
            }

            // Add event listeners for total amount updates
            document.addEventListener('DOMContentLoaded', function() {
                const allInputs = document.querySelectorAll('input[name*="quantity"], input[name*="unit_price"]');
                allInputs.forEach(input => {
                    input.addEventListener('change', updateTotalAmount);
                    input.addEventListener('input', updateTotalAmount);
                });
                updateTotalAmount();
                document.getElementById('requisition-form').reset();
                document.getElementById('recipient_id').value = '';

                const recipientTypeRadios = document.querySelectorAll('input[name="recipient_type"]');
                const internalContainer = document.getElementById('internal-recipient-container');
                const externalContainer = document.getElementById('external-recipient-container');
                const internalRecipientHelperParagraph = document.getElementById('selected-internal-recipient');
                const internalRecipientHelperName = document.getElementById('selected-internal-recipient-name');

                function refreshInternalRecipientHelper() {
                    if (!internalRecipientHelperParagraph || !internalRecipientHelperName) {
                        return;
                    }
                    const selectedType = document.querySelector('input[name="recipient_type"]:checked')?.value || 'internal';
                    if (selectedType !== 'internal') {
                        internalRecipientHelperParagraph.style.display = 'none';
                        internalRecipientHelperName.textContent = '—';
                        return;
                    }
                    internalRecipientHelperParagraph.style.display = 'block';
                    const label = getInternalRecipientLabel();
                    internalRecipientHelperName.textContent = label || '—';
                }

                function toggleRecipientFields() {
                    const selectedType = document.querySelector('input[name="recipient_type"]:checked').value;
                    if (selectedType === 'internal') {
                        internalContainer.style.display = 'block';
                        externalContainer.style.display = 'none';
                        // Remove required from external fields and add to internal
                        document.getElementById('external_recipient_name').removeAttribute('required');
                        document.getElementById('external_recipient_phone').removeAttribute('required');
                        document.getElementById('recipient_id').setAttribute('required', 'required');
                    } else {
                        internalContainer.style.display = 'none';
                        externalContainer.style.display = 'block';
                        // Remove required from internal field and add to external
                        document.getElementById('recipient_id').removeAttribute('required');
                        document.getElementById('external_recipient_name').setAttribute('required', 'required');
                        document.getElementById('external_recipient_phone').setAttribute('required', 'required');
                    }
                    refreshInternalRecipientHelper();
                }

                recipientTypeRadios.forEach(radio => {
                    radio.addEventListener('change', toggleRecipientFields);
                });
                const recipientSelectElement = document.getElementById('recipient_id');
                if (recipientSelectElement) {
                    recipientSelectElement.addEventListener('change', refreshInternalRecipientHelper);
                }
                refreshInternalRecipientHelper();
                toggleRecipientFields();
            });

            window.onload = function() {
                console.log('DEBUG: Page loaded at ' + new Date().toLocaleString() + ', resetting form');
                updateTotalAmount();
            };
        </script>

        <noscript>
            <p style="color: #FF0000; text-align: center; font-size: 16px;">JavaScript est désactivé. Veuillez activer JavaScript pour utiliser ce formulaire.</p>
        </noscript>

        <?php if ($role_name === 'Accountant' && !empty($requisitions)): ?>
            <h3 style="color: #4B2F1F; margin-top: 20px; margin-bottom: 10px;">Traiter Réquisitions Approuvées</h3>
            <table style="width: 100%; border-collapse: collapse; margin-top: 10px;">
                <thead>
                    <tr style="background-color: #4B2F1F; color: #F4F0E4;">
                        <th style="padding: 10px; border: 1px solid #4B2F1F;">ID</th>
                        <th style="padding: 10px; border: 1px solid #4B2F1F;">Initiateur</th>
                        <th style="padding: 10px; border: 1px solid #4B2F1F;">Destinataire</th>
                        <th style="padding: 10px; border: 1px solid #4B2F1F;">Montant (BIF)</th>
                        <th style="padding: 10px; border: 1px solid #4B2F1F;">Action</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($requisitions as $req): ?>
                        <tr style="background-color: #FFFFFF;">
                            <td style="padding: 10px; border: 1px solid #4B2F1F;"><?php echo htmlspecialchars($req['id']); ?></td>
                            <td style="padding: 10px; border: 1px solid #4B2F1F;"><?php echo htmlspecialchars($req['initiator_name']); ?></td>
                            <td style="padding: 10px; border: 1px solid #4B2F1F;"><?php echo htmlspecialchars($req['recipient_display'] ?? $req['recipient_name']); ?></td>
                            <td style="padding: 10px; border: 1px solid #4B2F1F;"><?php echo number_format($req['total_amount'], 0, ',', '.'); ?> BIF</td>
                            <td style="padding: 10px; border: 1px solid #4B2F1F;">
                                <form method="POST" style="margin: 0;">
                                    <input type="hidden" name="requisition_id" value="<?php echo htmlspecialchars($req['id']); ?>">
                                    <div style="margin-bottom: 10px;">
                                        <label for="cashier_id_<?php echo htmlspecialchars($req['id']); ?>" style="color: #4B2F1F;">Caissier:</label>
                                        <select name="cashier_id" id="cashier_id_<?php echo htmlspecialchars($req['id']); ?>" required style="padding: 5px; border-radius: 5px; border: 1px solid #4B2F1F;">
                                            <option value="">Sélectionner Caissier</option>
                                            <?php foreach ($cashiers as $cashier): ?>
                                                <option value="<?php echo htmlspecialchars($cashier['id']); ?>"><?php echo htmlspecialchars($cashier['full_name']); ?></option>
                                            <?php endforeach; ?>
                                        </select>
                                    </div>
                                    <div style="margin-bottom: 10px;">
                                        <label for="receipt_number_<?php echo htmlspecialchars($req['id']); ?>" style="color: #4B2F1F;">Numéro de Reçu:</label>
                                        <input type="text" name="receipt_number" id="receipt_number_<?php echo htmlspecialchars($req['id']); ?>" required style="padding: 5px; border-radius: 5px; border: 1px solid #4B2F1F;">
                                    </div>
                                    <div style="margin-bottom: 10px;">
                                        <label for="recipient_signature_<?php echo htmlspecialchars($req['id']); ?>" style="color: #4B2F1F;">Signature Destinataire:</label>
                                        <input type="text" name="recipient_signature" id="recipient_signature_<?php echo htmlspecialchars($req['id']); ?>" required style="padding: 5px; border-radius: 5px; border: 1px solid #4B2F1F;">
                                    </div>
                                    <button type="submit" name="process_payment" style="background-color: #4B2F1F; color: #F4F0E4; padding: 5px 10px; border: none; border-radius: 5px; cursor: pointer;">Traiter Paiement</button>
                                </form>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        <?php elseif ($role_name === 'Accountant'): ?>
            <p style="color: #000000; font-size: 16px;">Aucune réquisition approuvée disponible pour paiement.</p>
        <?php endif; ?>
    </div>
</div>
<?php include $footer_path ?: "</body></html>"; ?>
